"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.replaceRefreshableAlertFields = exports.replaceEmptyAlertFields = exports.removeUnflattenedFieldsFromAlert = exports.expandFlattenedAlert = exports.compactObject = void 0;
var _lodash = require("lodash");
var _alert_conflict_resolver = require("./alert_conflict_resolver");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const expandDottedField = (dottedFieldName, val) => {
  const parts = dottedFieldName.split('.');
  if (parts.length === 1) {
    return {
      [parts[0]]: val
    };
  } else {
    return {
      [parts[0]]: expandDottedField(parts.slice(1).join('.'), val)
    };
  }
};
const expandFlattenedAlert = alert => {
  return Object.entries(alert).reduce((acc, [key, val]) => (0, _lodash.merge)(acc, expandDottedField(key, val)), {});
};
exports.expandFlattenedAlert = expandFlattenedAlert;
// Removes empty nested objects
const compactObject = obj => {
  return Object.keys(obj).filter(key => {
    // just filter out empty objects
    // keep any primitives or arrays, even empty arrays
    return !(0, _lodash.isUndefined)(obj[key]) && (Array.isArray(obj[key]) || typeof obj[key] !== 'object' || typeof obj[key] === 'object' && (!(0, _lodash.isEmpty)(obj[key]) || obj[key] === null));
  }).reduce((acc, curr) => {
    if (typeof obj[curr] !== 'object' || Array.isArray(obj[curr])) {
      acc[curr] = obj[curr];
    } else if ((0, _lodash.isNull)(obj[curr])) {
      acc[curr] = null;
    } else {
      const compacted = compactObject(obj[curr]);
      if (!(0, _lodash.isEmpty)(compacted)) {
        acc[curr] = compacted;
      }
    }
    return acc;
  }, {});
};

/**
 * If we're replacing field values in an unflattened alert
 * with the flattened version, we want to remove the unflattened version
 * to avoid duplicate data in the doc
 */
exports.compactObject = compactObject;
const removeUnflattenedFieldsFromAlert = (alert, flattenedData) => {
  // make a copy of the alert
  let alertCopy = (0, _lodash.cloneDeep)(alert);

  // for each flattened field in the flattened data object,
  // check whether that path exists in the unflattened alert
  // and omit it if it does
  Object.keys(flattenedData).forEach(payloadKey => {
    const val = (0, _lodash.get)(alertCopy, payloadKey, null);
    if (null == alertCopy[payloadKey] && null != val) {
      alertCopy = (0, _lodash.omit)(alertCopy, payloadKey);
    }
  });
  return compactObject(alertCopy);
};
exports.removeUnflattenedFieldsFromAlert = removeUnflattenedFieldsFromAlert;
const replaceRefreshableAlertFields = alert => {
  // Make sure that any alert fields that are updateable are flattened.
  return _alert_conflict_resolver.REFRESH_FIELDS_ALL.reduce((acc, currField) => {
    const value = (0, _lodash.get)(alert, currField);
    if (null != value) {
      acc[currField] = value;
    }
    return acc;
  }, {});
};
exports.replaceRefreshableAlertFields = replaceRefreshableAlertFields;
const replaceEmptyAlertFields = (alert, payload = {}) => {
  Object.keys(alert).forEach(key => {
    const value = (0, _lodash.get)(payload, key, null);
    if ((0, _lodash.isNull)(value) && Array.isArray(alert[key])) {
      payload[key] = [];
    }
  });
  return payload;
};
exports.replaceEmptyAlertFields = replaceEmptyAlertFields;