"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useIsWithinBreakpoints = void 0;
exports.useIsWithinMaxBreakpoint = useIsWithinMaxBreakpoint;
exports.useIsWithinMinBreakpoint = useIsWithinMinBreakpoint;
var _theme = require("../theme");
var _current_breakpoint_hook = require("./current_breakpoint_hook");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Given an array of breakpoint keys, this hook returns true or false
 * if the breakpoint size of the current window width falls within
 * any of the named breakpoints.
 *
 * @param {EuiThemeBreakpoint[]} sizes An array of named EUI breakpoints
 * @param {boolean} isResponsive Some components have the option to turn off responsive behavior.
 *   Since hooks can't be called conditionally, it's easier to pass the condition into the hook
 * @returns {boolean} Returns `true` if current breakpoint name is included in `sizes`
 */
var useIsWithinBreakpoints = exports.useIsWithinBreakpoints = function useIsWithinBreakpoints(sizes) {
  var isResponsive = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;
  var currentBreakpoint = (0, _current_breakpoint_hook.useCurrentEuiBreakpoint)();
  return currentBreakpoint && isResponsive ? sizes.includes(currentBreakpoint) : false;
};

/**
 * Given a max breakpoint key, this hook returns true if the breakpoint size
 * of the current window width falls within the max breakpoint or any below,
 * and false otherwise
 *
 * @param {EuiThemeBreakpoint} max The named max breakpoint to check against
 * @returns {boolean} Will return `false` if it can't find a value for the `max` breakpoint
 */
function useIsWithinMaxBreakpoint(max) {
  var _useEuiTheme = (0, _theme.useEuiTheme)(),
    breakpoints = _useEuiTheme.euiTheme.breakpoint;
  var currentBreakpoint = (0, _current_breakpoint_hook.useCurrentEuiBreakpoint)();
  if (currentBreakpoint == null || breakpoints[max] == null) {
    return false;
  }
  return breakpoints[currentBreakpoint] <= breakpoints[max];
}

/**
 * Given a min breakpoint key, this hook returns true if the breakpoint size
 * of the current window width falls within the min breakpoint or any above,
 * and false otherwise
 *
 * @param {EuiThemeBreakpoint} min The named min breakpoint to check against
 * @returns {boolean} Will return `false` if it can't find a value for the `min` breakpoint
 */
function useIsWithinMinBreakpoint(min) {
  var _useEuiTheme2 = (0, _theme.useEuiTheme)(),
    breakpoints = _useEuiTheme2.euiTheme.breakpoint;
  var currentBreakpoint = (0, _current_breakpoint_hook.useCurrentEuiBreakpoint)();
  if (currentBreakpoint == null || breakpoints[min] == null) {
    return false;
  }
  return breakpoints[currentBreakpoint] >= breakpoints[min];
}