"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFlyoutActivityStage = void 0;
var _react = require("react");
var _const = require("./const");
var _hooks = require("./hooks");
var _actions = require("./actions");
var _provider = require("./provider");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Encapsulates all activity-stage transitions and animation-driven updates
 * for managed flyouts.
 */
var useFlyoutActivityStage = exports.useFlyoutActivityStage = function useFlyoutActivityStage(_ref) {
  var _ctx$state$flyouts$fi;
  var flyoutId = _ref.flyoutId,
    level = _ref.level;
  var isActive = (0, _hooks.useIsFlyoutActive)(flyoutId);
  var hasChild = (0, _hooks.useHasChildFlyout)(flyoutId);
  var layoutMode = (0, _hooks.useFlyoutLayoutMode)();
  var ctx = (0, _provider.useFlyoutManager)();
  var stage = (ctx === null || ctx === void 0 || (_ctx$state$flyouts$fi = ctx.state.flyouts.find(function (f) {
    return f.flyoutId === flyoutId;
  })) === null || _ctx$state$flyouts$fi === void 0 ? void 0 : _ctx$state$flyouts$fi.activityStage) || (isActive ? _const.STAGE_ACTIVE : _const.STAGE_INACTIVE);
  var stageRef = (0, _react.useRef)(stage);
  if (stageRef.current !== stage) {
    stageRef.current = stage;
  }

  /**
   * 1. ACTIVE -> CLOSING when no longer the active flyout.
   * 2. INACTIVE -> RETURNING when it becomes active again (e.g., reopened or brought forward).
   * 3. (Main flyout only) ACTIVE + stacked + has child -> BACKGROUNDING (begin background animation).
   * 4. (Main only) BACKGROUNDED/BACKGROUNDING + (child gone OR side-by-side) -> RETURNING (bring main to foreground).
   *
   * Any stages that depend on animation end (OPENING, RETURNING, CLOSING, BACKGROUNDING) are finalized in `onAnimationEnd`.
   */
  (0, _react.useEffect)(function () {
    var s = stageRef.current;
    var next = null;
    if (s === _const.STAGE_ACTIVE && !isActive) next = _const.STAGE_CLOSING;else if (s === _const.STAGE_INACTIVE && isActive) {
      next = _const.STAGE_RETURNING;
    } else if (level === _const.LEVEL_MAIN && isActive && s === _const.STAGE_ACTIVE && hasChild && layoutMode === _const.LAYOUT_MODE_STACKED) next = _const.STAGE_BACKGROUNDING;else if (level === _const.LEVEL_MAIN && (s === _const.STAGE_BACKGROUNDED || s === _const.STAGE_BACKGROUNDING) && (!hasChild || layoutMode === _const.LAYOUT_MODE_SIDE_BY_SIDE)) next = _const.STAGE_RETURNING;
    if (next && next !== s) {
      var _ctx$dispatch;
      ctx === null || ctx === void 0 || (_ctx$dispatch = ctx.dispatch) === null || _ctx$dispatch === void 0 || _ctx$dispatch.call(ctx, (0, _actions.setActivityStage)(flyoutId, next));
      stageRef.current = next;
    }
  }, [isActive, hasChild, layoutMode, level, ctx, flyoutId, stage]);

  /**
   * Get the stage to transition to for given current stage.
   * Returns `null` if stage should remain unchanged.
   *
   * Stage transitions:
   *  - OPENING / RETURNING -> ACTIVE
   *  - CLOSING -> INACTIVE
   *  - BACKGROUNDING -> BACKGROUNDED
   */
  var getNextStage = function getNextStage(stage) {
    switch (stage) {
      case _const.STAGE_OPENING:
      case _const.STAGE_RETURNING:
        return _const.STAGE_ACTIVE;
      case _const.STAGE_CLOSING:
        return _const.STAGE_INACTIVE;
      case _const.STAGE_BACKGROUNDING:
        return _const.STAGE_BACKGROUNDED;
    }
    return null;
  };

  /**
   * onAnimationEnd event handler that must be passed to EuiFlyout.
   * It handles transitions between stages and updates activity stage
   * in EuiFlyoutManagerContext.
   */
  var onAnimationEnd = (0, _react.useCallback)(function () {
    var currentStage = stageRef.current;
    var nextStage = getNextStage(currentStage);
    if (nextStage && nextStage !== currentStage) {
      var _ctx$dispatch2;
      ctx === null || ctx === void 0 || (_ctx$dispatch2 = ctx.dispatch) === null || _ctx$dispatch2 === void 0 || _ctx$dispatch2.call(ctx, (0, _actions.setActivityStage)(flyoutId, nextStage));
      stageRef.current = nextStage;
    }
  }, [ctx, flyoutId]);
  return {
    activityStage: stage,
    onAnimationEnd: onAnimationEnd
  };
};