"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEndpointHostIsolationStatus = void 0;
var _lodash = require("lodash");
var _react = require("react");
var _api = require("../../../../../detections/containers/detection_engine/alerts/api");
var _endpoint_pending_actions = require("../../../../lib/endpoint/endpoint_pending_actions");
var _validators = require("../../../../utils/validators");
var _types = require("../../../../../../common/endpoint/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves the current isolation status of a host and the agent/host status
 */
const useEndpointHostIsolationStatus = ({
  agentId,
  agentType
}) => {
  const [isIsolated, setIsIsolated] = (0, _react.useState)(false);
  const [capabilities, setCapabilities] = (0, _react.useState)([]);
  const [agentStatus, setAgentStatus] = (0, _react.useState)();
  const [pendingIsolation, setPendingIsolation] = (0, _react.useState)(0);
  const [pendingUnisolation, setPendingUnisolation] = (0, _react.useState)(0);
  const [loading, setLoading] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    const abortCtrl = new AbortController();
    // isMounted tracks if a component is mounted before changing state
    let isMounted = true;
    let fleetAgentId;
    setLoading(true);
    const fetchData = async () => {
      try {
        const metadataResponse = await (0, _api.getHostMetadata)({
          agentId,
          signal: abortCtrl.signal
        });
        if (isMounted) {
          setIsIsolated((0, _validators.isEndpointHostIsolated)(metadataResponse.metadata));
          if (metadataResponse.metadata.Endpoint.capabilities) {
            setCapabilities([...metadataResponse.metadata.Endpoint.capabilities]);
          }
          setAgentStatus(metadataResponse.host_status);
          fleetAgentId = metadataResponse.metadata.elastic.agent.id;
        }
      } catch (error) {
        // don't show self-aborted requests errors to the user
        if (error.name === 'AbortError') {
          return;
        }
        if (isMounted && error.body.statusCode === 400) {
          setAgentStatus(_types.HostStatus.UNENROLLED);
        }
      }
      if (!(fleetAgentId && fleetAgentId.length)) {
        return;
      }
      try {
        const {
          data
        } = await (0, _endpoint_pending_actions.fetchPendingActionsByAgentId)(fleetAgentId);
        if (isMounted) {
          var _data$0$pending_actio, _data$0$pending_actio2, _data$0$pending_actio3, _data$0$pending_actio4;
          setPendingIsolation((_data$0$pending_actio = (_data$0$pending_actio2 = data[0].pending_actions) === null || _data$0$pending_actio2 === void 0 ? void 0 : _data$0$pending_actio2.isolate) !== null && _data$0$pending_actio !== void 0 ? _data$0$pending_actio : 0);
          setPendingUnisolation((_data$0$pending_actio3 = (_data$0$pending_actio4 = data[0].pending_actions) === null || _data$0$pending_actio4 === void 0 ? void 0 : _data$0$pending_actio4.unisolate) !== null && _data$0$pending_actio3 !== void 0 ? _data$0$pending_actio3 : 0);
        }
      } catch (error) {
        // silently catch non-user initiated error
      }
    };
    if (!(0, _lodash.isEmpty)(agentId) && agentType === 'endpoint') {
      fetchData().finally(() => {
        if (isMounted) {
          setLoading(false);
        }
      });
    }
    return () => {
      // updates to show component is unmounted
      isMounted = false;
      abortCtrl.abort();
    };
  }, [agentId, agentType]);
  return {
    loading,
    capabilities,
    isIsolated,
    agentStatus,
    pendingIsolation,
    pendingUnisolation
  };
};
exports.useEndpointHostIsolationStatus = useEndpointHostIsolationStatus;