"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QradarRulesXmlParser = void 0;
var _xml2js = _interopRequireDefault(require("xml2js"));
var _xml = require("../xml/xml");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class QradarRulesXmlParser extends _xml.XmlParser {
  async processRuleXml(qradarRule) {
    var _this$findDeepValue;
    const ruleData = this.getStrValue(this.findDeep(qradarRule, 'rule_data'));
    let decodedRuleData;
    try {
      decodedRuleData = Buffer.from(ruleData, 'base64').toString('utf-8');
      // Sanitize HTML content within <text> tags
      decodedRuleData = this.sanitizeTextTagsInRuleData(decodedRuleData);
    } catch (error) {
      throw new Error(`Failed to decode rule_data from base64: ${error.message}`);
    }
    const parsedRuleData = await this.parseRuleData(decodedRuleData);
    const id = this.findDeepValue(parsedRuleData, 'rule', 'id');
    const name = this.findDeep(parsedRuleData, 'name');
    const notes = this.findDeep(parsedRuleData, 'notes');
    const isBuildingBlockVal = (_this$findDeepValue = this.findDeepValue(parsedRuleData, 'rule', 'buildingBlock')) !== null && _this$findDeepValue !== void 0 ? _this$findDeepValue : 'false';
    if (name && notes && isBuildingBlockVal) {
      const title = this.getStrValue(name);
      const description = this.getStrValue(notes);
      const isBuildingBlock = isBuildingBlockVal === 'true';
      return {
        id: id,
        title,
        description,
        rule_type: isBuildingBlock ? 'building_block' : 'default',
        rule_data: decodedRuleData
      };
    }
  }
  async getRules() {
    const parsedXml = await this.parse();
    const rules = this.findDeep(parsedXml, 'custom_rule');
    const qradarRules = Array.isArray(rules) ? rules : [];
    const processQradarRulePromises = qradarRules.map(this.processRuleXml.bind(this));
    const parsedRules = await Promise.all(processQradarRulePromises);
    return parsedRules.filter(Boolean);
  }
  async parseRuleData(ruleData) {
    return _xml2js.default.parseStringPromise(ruleData, {
      explicitArray: true
    });
  }
  async parseSeverityFromRuleData(ruleData) {
    const parsedRuleData = await this.parseRuleData(ruleData);
    return this.findDeepValue(parsedRuleData, 'newevent', 'severity');
  }
  async getResources() {
    const parsedXml = await this.parse();
    const [sensordevicetypes] = await Promise.all([this.getSensortDeviceType(parsedXml)]);
    return {
      qidmap: undefined,
      reference_data_rules: undefined,
      sensordevicetype: sensordevicetypes,
      sensordeviceprotocols: undefined,
      sensordevicecategory: undefined,
      ariel_property_expression: undefined,
      ariel_regex_property: undefined,
      reference_data: undefined,
      offense_type: undefined
    };
  }
  async getSensortDeviceType(parsedXml) {
    var _ref;
    const sensordevicetypes = this.findDeep(parsedXml, 'sensordevicetype');
    return ((_ref = Array.isArray(sensordevicetypes) ? sensordevicetypes : []) === null || _ref === void 0 ? void 0 : _ref.map(deviceType => ({
      name: this.getStrValue(deviceType.devicetypename),
      description: this.getStrValue(deviceType.devicetypedescription),
      content: JSON.stringify(deviceType)
    }))) || [];
  }
  getStrValue(val) {
    if (Array.isArray(val)) {
      return val[0].trim();
    }
    return val.trim();
  }

  /**
   * Sanitizes text content by decoding HTML entities and removing HTML tags.
   * QRadar rule text elements can contain HTML like:
   * `when &lt;a href='javascript:...'&gt;any&lt;/a&gt; of &lt;a&gt;Reference Set Name&lt;/a&gt;`
   *
   * This method converts it to plain text:
   * `when any of Reference Set Name`
   *
   * @param text - The text content that may contain HTML entities and tags
   * @returns Sanitized plain text
   */
  sanitizeHtmlText(text) {
    // First, decode common HTML entities
    let decoded = text.replace(/&lt;/g, '<').replace(/&gt;/g, '>').replace(/&amp;/g, '&').replace(/&quot;/g, '"').replace(/&#39;/g, "'").replace(/&apos;/g, "'");

    // Remove all HTML tags and keep only the text content
    // This regex matches opening tags, closing tags, and self-closing tags
    decoded = decoded.replace(/<[^>]*>/g, '');

    // Clean up any extra whitespace that might result from tag removal
    decoded = decoded.replace(/\s+/g, ' ').trim();
    return decoded;
  }

  /**
   * Sanitizes the content within <text> tags in the rule data XML.
   * This cleans up HTML entities and tags within test text elements
   * so the stored rule_data is clean.
   *
   * @param ruleData - The decoded XML rule data string
   * @returns The rule data with sanitized text content
   */
  sanitizeTextTagsInRuleData(ruleData) {
    // Find all <text>...</text> patterns and sanitize their content
    return ruleData.replace(/<text>([\s\S]*?)<\/text>/g, (match, content) => {
      const sanitizedContent = this.sanitizeHtmlText(content);
      return `<text>${sanitizedContent}</text>`;
    });
  }

  /**
   * Extracts the reference set name from a "Name - Type" format.
   * QRadar reference set names can include a type suffix like "AlphaNumeric" or "IP".
   * For example: "FireEye Whitelists - AlphaNumeric" -> "FireEye Whitelists"
   *
   * If no " - " separator is found, returns the original name.
   *
   * @param fullName - The full reference set name that may include type suffix
   * @returns The reference set name without the type suffix
   */
  extractReferenceSetName(fullName) {
    // Split by " - " (space-dash-space) to separate name from type
    const separatorIndex = fullName.lastIndexOf(' - ');
    if (separatorIndex === -1) {
      // No separator found, return the full name
      return fullName;
    }

    // Return the name part (everything before the last " - ")
    return fullName.substring(0, separatorIndex).trim();
  }

  /**
   * Extracts reference set names from QRadar rule data XML.
   * Reference sets are identified by ReferenceSetTest tests, and their names
   * are extracted from text patterns like "contained in any of Name1, Name2"
   * or "contained in all of Name1, Name2".
   *
   * @param ruleData - The decoded XML rule data string
   * @returns Array of unique reference set names found in the rule
   */
  async getReferenceSetsFromRuleData(ruleData) {
    const parsedRuleData = await this.parseRuleData(ruleData);

    // Find all test elements in the rule data
    const tests = this.findAllDeep(parsedRuleData, 'test');
    const referenceSetNames = [];
    for (const test of tests) {
      var _test$$;
      // Check if this is a ReferenceSetTest by looking at the 'name' attribute
      const testName = (_test$$ = test.$) === null || _test$$ === void 0 ? void 0 : _test$$.name;
      if (testName === 'com.q1labs.semsources.cre.tests.ReferenceSetTest') {
        // Extract the text element which contains the reference set names
        const textElement = this.findDeep(test, 'text');
        if (textElement) {
          const rawTextContent = this.getStrValue(textElement);
          // Sanitize the text content to remove HTML tags and decode entities
          const textContent = this.sanitizeHtmlText(rawTextContent);

          // Parse the pattern: "contained in any/all of Name1, Name2, Name3"
          // The pattern can be either "contained in any of" or "contained in all of"
          const match = textContent.match(/contained in (?:any|all).* of (.+)/);
          if (match && match[1]) {
            // Split by comma, clean up each name, and extract the reference set name
            // (removing type suffix like "- AlphaNumeric" or "- IP")
            const names = match[1].split(',').map(name => name.trim()).filter(name => name.length > 0).map(name => this.extractReferenceSetName(name));
            referenceSetNames.push(...names);
          }
        }
      }
    }

    // Return unique names only
    return [...new Set(referenceSetNames)];
  }
}
exports.QradarRulesXmlParser = QradarRulesXmlParser;