"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.policyFactoryWithoutPaidFeatures = exports.policyFactoryWithoutPaidEnterpriseFeatures = exports.policyFactoryWithSupportedFeatures = exports.policyFactory = exports.DefaultPolicyRuleNotificationMessage = exports.DefaultPolicyNotificationMessage = exports.DefaultPolicyDeviceNotificationMessage = void 0;
var _types = require("../types");
var _policy_config_helpers = require("./policy_config_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Return a new default `PolicyConfig` for platinum and above licenses
 */
const policyFactory = ({
  license = '',
  cloud = false,
  licenseUuid = '',
  clusterUuid = '',
  clusterName = '',
  serverless = false,
  isGlobalTelemetryEnabled = false
} = {}) => {
  const policy = {
    meta: {
      license,
      license_uuid: licenseUuid,
      cluster_uuid: clusterUuid,
      cluster_name: clusterName,
      cloud,
      serverless
    },
    global_manifest_version: 'latest',
    global_telemetry_enabled: isGlobalTelemetryEnabled,
    windows: {
      events: {
        credential_access: true,
        dll_and_driver_load: true,
        dns: true,
        file: true,
        network: true,
        process: true,
        registry: true,
        security: true
      },
      device_control: {
        enabled: true,
        usb_storage: _types.DeviceControlAccessLevel.deny_all
      },
      malware: {
        mode: _types.ProtectionModes.prevent,
        blocklist: true,
        on_write_scan: true
      },
      ransomware: {
        mode: _types.ProtectionModes.prevent,
        supported: true
      },
      memory_protection: {
        mode: _types.ProtectionModes.prevent,
        supported: true
      },
      behavior_protection: {
        mode: _types.ProtectionModes.prevent,
        reputation_service: cloud,
        // Defaults to true if on cloud
        supported: true
      },
      popup: {
        malware: {
          message: '',
          enabled: true
        },
        ransomware: {
          message: '',
          enabled: true
        },
        memory_protection: {
          message: '',
          enabled: true
        },
        behavior_protection: {
          message: '',
          enabled: true
        },
        device_control: {
          message: '',
          enabled: true
        }
      },
      logging: {
        file: 'info'
      },
      antivirus_registration: {
        mode: _types.AntivirusRegistrationModes.sync,
        enabled: true // Defaults to true since Malware protection is set to prevent and mode is set to sync
      },
      attack_surface_reduction: {
        credential_hardening: {
          enabled: true
        }
      }
    },
    mac: {
      events: {
        dns: true,
        process: true,
        file: true,
        network: true,
        security: true
      },
      malware: {
        mode: _types.ProtectionModes.prevent,
        blocklist: true,
        on_write_scan: true
      },
      device_control: {
        enabled: true,
        usb_storage: _types.DeviceControlAccessLevel.deny_all
      },
      behavior_protection: {
        mode: _types.ProtectionModes.prevent,
        reputation_service: cloud,
        // Defaults to true if on cloud
        supported: true
      },
      memory_protection: {
        mode: _types.ProtectionModes.prevent,
        supported: true
      },
      popup: {
        malware: {
          message: '',
          enabled: true
        },
        behavior_protection: {
          message: '',
          enabled: true
        },
        memory_protection: {
          message: '',
          enabled: true
        },
        device_control: {
          message: '',
          enabled: true
        }
      },
      logging: {
        file: 'info'
      },
      advanced: {
        capture_env_vars: 'DYLD_INSERT_LIBRARIES,DYLD_FRAMEWORK_PATH,DYLD_LIBRARY_PATH,LD_PRELOAD'
      }
    },
    linux: {
      events: {
        dns: true,
        file: true,
        process: true,
        network: true,
        session_data: false,
        tty_io: false
      },
      malware: {
        mode: _types.ProtectionModes.prevent,
        blocklist: true,
        on_write_scan: true
      },
      behavior_protection: {
        mode: _types.ProtectionModes.prevent,
        reputation_service: cloud,
        // Defaults to true if on cloud
        supported: true
      },
      memory_protection: {
        mode: _types.ProtectionModes.prevent,
        supported: true
      },
      popup: {
        malware: {
          message: '',
          enabled: true
        },
        behavior_protection: {
          message: '',
          enabled: true
        },
        memory_protection: {
          message: '',
          enabled: true
        }
      },
      logging: {
        file: 'info'
      },
      advanced: {
        capture_env_vars: 'LD_PRELOAD,LD_LIBRARY_PATH'
      }
    }
  };
  policy.meta.billable = (0, _policy_config_helpers.isBillablePolicy)(policy);
  return policy;
};

/**
 * Strips paid features from an existing or new `PolicyConfig` for license below enterprise
 */
exports.policyFactory = policyFactory;
const policyFactoryWithoutPaidEnterpriseFeatures = (policy = policyFactory()) => {
  return {
    ...policy,
    global_manifest_version: 'latest',
    windows: {
      ...policy.windows,
      device_control: {
        enabled: false,
        usb_storage: _types.DeviceControlAccessLevel.audit
      },
      popup: {
        ...policy.windows.popup,
        device_control: {
          enabled: false,
          message: ''
        }
      }
    },
    mac: {
      ...policy.mac,
      device_control: {
        enabled: false,
        usb_storage: _types.DeviceControlAccessLevel.audit
      },
      popup: {
        ...policy.mac.popup,
        device_control: {
          enabled: false,
          message: ''
        }
      }
    }
  };
};

/**
 * Strips paid features from an existing or new `PolicyConfig` for gold and below license
 */
exports.policyFactoryWithoutPaidEnterpriseFeatures = policyFactoryWithoutPaidEnterpriseFeatures;
const policyFactoryWithoutPaidFeatures = (policy = policyFactory()) => {
  const rollbackConfig = {
    rollback: {
      self_healing: {
        enabled: false
      }
    }
  };
  return {
    ...policy,
    global_manifest_version: 'latest',
    windows: {
      ...policy.windows,
      advanced: policy.windows.advanced === undefined ? undefined : {
        ...policy.windows.advanced,
        alerts: policy.windows.advanced.alerts === undefined ? {
          ...rollbackConfig
        } : {
          ...policy.windows.advanced.alerts,
          ...rollbackConfig
        }
      },
      ransomware: {
        mode: _types.ProtectionModes.off,
        supported: false
      },
      memory_protection: {
        mode: _types.ProtectionModes.off,
        supported: false
      },
      behavior_protection: {
        mode: _types.ProtectionModes.off,
        reputation_service: false,
        supported: false
      },
      attack_surface_reduction: {
        credential_hardening: {
          enabled: false
        }
      },
      device_control: {
        enabled: false,
        usb_storage: _types.DeviceControlAccessLevel.audit
      },
      popup: {
        ...policy.windows.popup,
        malware: {
          message: '',
          enabled: true // disabling/configuring malware popup is a paid feature
        },
        ransomware: {
          message: '',
          enabled: false
        },
        memory_protection: {
          message: '',
          enabled: false
        },
        behavior_protection: {
          message: '',
          enabled: false
        },
        device_control: {
          message: '',
          enabled: false
        }
      }
    },
    mac: {
      ...policy.mac,
      behavior_protection: {
        mode: _types.ProtectionModes.off,
        reputation_service: false,
        supported: false
      },
      memory_protection: {
        mode: _types.ProtectionModes.off,
        supported: false
      },
      device_control: {
        enabled: false,
        usb_storage: _types.DeviceControlAccessLevel.audit
      },
      popup: {
        ...policy.mac.popup,
        malware: {
          message: '',
          enabled: true // disabling/configuring malware popup is a paid feature
        },
        memory_protection: {
          message: '',
          enabled: false
        },
        behavior_protection: {
          message: '',
          enabled: false
        },
        device_control: {
          message: '',
          enabled: false
        }
      }
    },
    linux: {
      ...policy.linux,
      behavior_protection: {
        mode: _types.ProtectionModes.off,
        reputation_service: false,
        supported: false
      },
      memory_protection: {
        mode: _types.ProtectionModes.off,
        supported: false
      },
      popup: {
        ...policy.linux.popup,
        malware: {
          message: '',
          enabled: true // disabling/configuring malware popup is a paid feature
        },
        memory_protection: {
          message: '',
          enabled: false
        },
        behavior_protection: {
          message: '',
          enabled: false
        }
      }
    }
  };
};

/**
 * Enables support for paid features for an existing or new `PolicyConfig` for platinum and above license
 */
exports.policyFactoryWithoutPaidFeatures = policyFactoryWithoutPaidFeatures;
const policyFactoryWithSupportedFeatures = (policy = policyFactory()) => {
  return {
    ...policy,
    windows: {
      ...policy.windows,
      ransomware: {
        ...policy.windows.ransomware,
        supported: true
      },
      memory_protection: {
        ...policy.windows.memory_protection,
        supported: true
      },
      behavior_protection: {
        ...policy.windows.behavior_protection,
        supported: true
      }
    },
    mac: {
      ...policy.mac,
      behavior_protection: {
        ...policy.windows.behavior_protection,
        supported: true
      },
      memory_protection: {
        ...policy.mac.memory_protection,
        supported: true
      }
    },
    linux: {
      ...policy.linux,
      behavior_protection: {
        ...policy.windows.behavior_protection,
        supported: true
      },
      memory_protection: {
        ...policy.linux.memory_protection,
        supported: true
      }
    }
  };
};

/**
 * Reflects what string the Endpoint will use when message field is default/empty
 */
exports.policyFactoryWithSupportedFeatures = policyFactoryWithSupportedFeatures;
const DefaultPolicyNotificationMessage = exports.DefaultPolicyNotificationMessage = 'Elastic Security {action} {filename}';
const DefaultPolicyRuleNotificationMessage = exports.DefaultPolicyRuleNotificationMessage = 'Elastic Security {action} {rule}';
const DefaultPolicyDeviceNotificationMessage = exports.DefaultPolicyDeviceNotificationMessage = 'Device Control activity was detected';