"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UserProfileAPIClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_DATAPATHS = 'avatar,userSettings';
class UserProfileAPIClient {
  constructor(http) {
    (0, _defineProperty2.default)(this, "internalDataUpdates$", new _rxjs.Subject());
    /**
     * Emits event whenever user profile is changed by the user.
     */
    (0, _defineProperty2.default)(this, "dataUpdates$", this.internalDataUpdates$.asObservable());
    (0, _defineProperty2.default)(this, "_userProfile$", new _rxjs.BehaviorSubject(null));
    (0, _defineProperty2.default)(this, "_enabled$", new _rxjs.BehaviorSubject(false));
    (0, _defineProperty2.default)(this, "_userProfileLoaded$", new _rxjs.BehaviorSubject(false));
    /** Observable of the current user profile data */
    (0, _defineProperty2.default)(this, "userProfile$", this._userProfile$.asObservable());
    (0, _defineProperty2.default)(this, "userProfileLoaded$", this._userProfileLoaded$.asObservable().pipe((0, _rxjs.distinctUntilChanged)()));
    (0, _defineProperty2.default)(this, "enabled$", void 0);
    this.http = http;
    this.enabled$ = this.userProfileLoaded$.pipe((0, _rxjs.skipWhile)(loaded => !loaded), (0, _rxjs.switchMap)(() => this._enabled$.asObservable()), (0, _rxjs.distinctUntilChanged)());
  }
  start() {
    // Fetch the user profile with default path to initialize the user profile observable.
    // This will also enable or not the user profile for the user by checking if we receive a 404 on this request.
    this.getCurrent({
      dataPath: DEFAULT_DATAPATHS
    }).catch(() => {
      // silently ignore the error
    });
  }

  /**
   * Retrieves the user profile of the current user. If the profile isn't available, e.g. for the anonymous users or
   * users authenticated via authenticating proxies, the `null` value is returned.
   * @param [params] Get current user profile operation parameters.
   * @param params.dataPath By default `getCurrent()` returns user information, but does not return any user data. The
   * optional "dataPath" parameter can be used to return personal data for this user.
   */
  getCurrent(params) {
    if (this.http.anonymousPaths.isAnonymous(window.location.pathname)) {
      this._enabled$.next(false);
      this._userProfileLoaded$.next(true);
      return Promise.reject(new Error('Unable to retrieve user profile for anonymous paths'));
    }
    return this.http.get('/internal/security/user_profile', {
      query: {
        dataPath: params === null || params === void 0 ? void 0 : params.dataPath
      }
    }).then(response => {
      var _response$data;
      const data = (_response$data = response === null || response === void 0 ? void 0 : response.data) !== null && _response$data !== void 0 ? _response$data : {};
      const updated = (0, _lodash.merge)(this._userProfile$.getValue(), data);
      this._userProfile$.next(updated);
      this._enabled$.next(true);
      this._userProfileLoaded$.next(true);
      return response;
    }).catch(err => {
      var _err$response;
      // If we receive a 404 on the request, it means there are no user profile for the user.
      const notFound = (err === null || err === void 0 ? void 0 : (_err$response = err.response) === null || _err$response === void 0 ? void 0 : _err$response.status) === 404;
      this._enabled$.next(notFound ? false : true);
      this._userProfileLoaded$.next(true);
      return Promise.reject(err);
    });
  }

  /**
   * Retrieves multiple user profiles by their identifiers.
   * @param params Bulk get operation parameters.
   * @param params.uids List of user profile identifiers.
   * @param params.dataPath By default Elasticsearch returns user information, but does not return any user data. The
   * optional "dataPath" parameter can be used to return personal data for the requested user profiles.
   */
  bulkGet(params) {
    return this.http.post('/internal/security/user_profile/_bulk_get', {
      // Convert `Set` with UIDs to an array to make it serializable.
      body: JSON.stringify({
        ...params,
        uids: [...params.uids]
      })
    });
  }

  /**
   * Suggests multiple user profiles by search criteria.
   *
   * Note: This endpoint is not provided out-of-the-box by the platform. You need to expose your own
   * version within your app. An example of how to do this can be found in:
   * `examples/user_profile_examples/server/plugin.ts`
   *
   * @param path Path to your app's suggest endpoint.
   * @param params Suggest operation parameters.
   * @param params.name Query string used to match name-related fields in user profiles. The
   * following fields are treated as name-related: username, full_name and email.
   * @param params.size Desired number of suggestions to return. The default value is 10.
   * @param params.dataPath By default, suggest API returns user information, but does not return
   * any user data. The optional "dataPath" parameter can be used to return personal data for this
   * user (within `kibana` namespace only).
   */
  suggest(path, params) {
    return this.http.post(path, {
      body: JSON.stringify(params)
    });
  }

  /**
   * Updates user profile data of the current user.
   * @param data Application data to be written (merged with existing data).
   */
  update(data) {
    // Optimistic update the user profile Observable.
    const previous = this._userProfile$.getValue();
    this._userProfile$.next(data);
    return this.http.post('/internal/security/user_profile/_data', {
      body: JSON.stringify(data)
    }).then(() => {
      this.internalDataUpdates$.next(data);
    }).catch(err => {
      // Revert the user profile data to the previous state.
      this._userProfile$.next(previous);
      return Promise.reject(err);
    });
  }

  /**
   * Updates user profile data of the current user.
   * @param data Application data to be written (merged with existing data).
   */
  partialUpdate(data) {
    const updated = (0, _lodash.merge)(this._userProfile$.getValue(), data);
    return this.update(updated);
  }
}
exports.UserProfileAPIClient = UserProfileAPIClient;