"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.populateIndex = void 0;
var _productDocCommon = require("@kbn/product-doc-common");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const populateIndex = async ({
  esClient,
  indexName,
  archive,
  legacySemanticText,
  log,
  elserInferenceId,
  abortController
}) => {
  log.debug(`Starting populating index ${indexName}`);
  try {
    const contentEntries = archive.getEntryPaths().filter(_productDocCommon.isArtifactContentFilePath);
    for (const entryPath of contentEntries) {
      log.debug(`Indexing content for entry ${entryPath}`);
      const contentBuffer = await archive.getEntryContent(entryPath);
      await indexContentFile({
        indexName,
        esClient,
        contentBuffer,
        legacySemanticText,
        elserInferenceId,
        abortController
      });
    }
    log.debug(`Done populating index ${indexName}`);
  } catch (e) {
    log.error(`Error while trying to populate index ${indexName}: ${e}`);
    throw e;
  }
};
exports.populateIndex = populateIndex;
const indexContentFile = async ({
  indexName,
  contentBuffer,
  esClient,
  legacySemanticText,
  elserInferenceId = _common.internalElserInferenceId,
  abortController
}) => {
  const fileContent = contentBuffer.toString('utf-8');
  const lines = fileContent.split('\n');
  const documents = lines.map(line => line.trim()).filter(line => line.length > 0).map(line => {
    return JSON.parse(line);
  }).map(doc => rewriteInferenceId({
    document: doc,
    inferenceId: elserInferenceId,
    legacySemanticText
  }));
  const operations = documents.flatMap(document => [{
    index: {
      _index: indexName
    }
  }, document]);
  const response = await esClient.bulk({
    refresh: false,
    operations
  }, {
    signal: abortController === null || abortController === void 0 ? void 0 : abortController.signal
  });
  if (response.errors) {
    var _response$items$find$, _response$items$find, _response$items$find$2;
    const error = (_response$items$find$ = (_response$items$find = response.items.find(item => {
      var _item$index;
      return (_item$index = item.index) === null || _item$index === void 0 ? void 0 : _item$index.error;
    })) === null || _response$items$find === void 0 ? void 0 : (_response$items$find$2 = _response$items$find.index) === null || _response$items$find$2 === void 0 ? void 0 : _response$items$find$2.error) !== null && _response$items$find$ !== void 0 ? _response$items$find$ : 'unknown error';
    throw new Error(`Error indexing documents: ${JSON.stringify(error)}`);
  }
};
const rewriteInferenceId = ({
  document,
  inferenceId,
  legacySemanticText
}) => {
  const semanticFieldsRoot = legacySemanticText ? document : document._inference_fields;
  // we don't need to handle nested fields, we don't have any and won't.
  Object.values(semanticFieldsRoot !== null && semanticFieldsRoot !== void 0 ? semanticFieldsRoot : {}).forEach(field => {
    if (field.inference) {
      field.inference.inference_id = inferenceId;
    }
  });
  return document;
};