"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ArtifactManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _productDocCommon = require("@kbn/product-doc-common");
var _fs = require("@kbn/fs");
var _common = require("../../../common");
var _semver = require("./utils/semver");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const mapDatasetSampleTypeToProduct = {
  [_common.DatasetSampleType.elasticsearch]: _productDocCommon.DocumentationProduct.elasticsearch
};
class ArtifactManager {
  constructor({
    artifactsFolder,
    artifactRepositoryUrl,
    kibanaVersion,
    logger
  }) {
    (0, _defineProperty2.default)(this, "artifactsFolder", void 0);
    (0, _defineProperty2.default)(this, "artifactRepositoryUrl", void 0);
    (0, _defineProperty2.default)(this, "currentVersion", void 0);
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "artifactVersions", void 0);
    (0, _defineProperty2.default)(this, "downloadedFiles", new Set());
    this.artifactsFolder = artifactsFolder;
    this.artifactRepositoryUrl = artifactRepositoryUrl;
    this.currentVersion = (0, _semver.majorMinor)(kibanaVersion);
    this.log = logger;
  }
  async prepareArtifact(sampleType, abortController) {
    const productName = mapDatasetSampleTypeToProduct[sampleType];
    if (!productName) {
      throw new Error(`Unsupported sample type for artifact preparation: ${sampleType}`);
    }
    const productVersion = (0, _semver.majorMinor)(await this.getProductVersion(productName));
    const artifactFileName = (0, _productDocCommon.getArtifactName)({
      productName,
      productVersion
    });
    const artifactUrl = `${this.artifactRepositoryUrl}/${artifactFileName}`;
    const artifactPathAtVolume = `${this.artifactsFolder}/${artifactFileName}`;
    this.log.debug(`Downloading artifact from [${artifactUrl}]`);
    const artifactFullPath = await (0, _utils.download)(artifactUrl, artifactPathAtVolume, 'application/zip', abortController);
    this.downloadedFiles.add(artifactPathAtVolume);
    const archive = await (0, _utils.openZipArchive)(artifactFullPath);
    (0, _utils.validateArtifactArchive)(archive);
    const [manifest, mappings] = await Promise.all([(0, _utils.loadManifestFile)(archive), (0, _utils.loadMappingFile)(archive)]);
    return {
      archive,
      manifest,
      mappings
    };
  }
  async getProductVersion(productName) {
    if (!this.artifactVersions) {
      this.artifactVersions = await (0, _utils.fetchArtifactVersions)({
        artifactRepositoryUrl: this.artifactRepositoryUrl
      });
    }
    const availableVersions = this.artifactVersions[productName];
    if (!(availableVersions !== null && availableVersions !== void 0 && availableVersions.length)) {
      throw new Error(`No versions found for product [${productName}]`);
    }
    return availableVersions.includes(this.currentVersion) ? this.currentVersion : (0, _semver.latestVersion)(availableVersions);
  }
  async cleanup() {
    for (const filePath of this.downloadedFiles) {
      try {
        await (0, _fs.deleteFile)(filePath);
        this.log.debug(`Deleted downloaded file: ${filePath}`);
      } catch (error) {
        this.log.warn(`Failed to delete file ${filePath}: ${error.message}`);
      }
    }
    this.downloadedFiles.clear();
  }
}
exports.ArtifactManager = ArtifactManager;