"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleUnavailable = exports.RequestHandler = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _reportingServer = require("@kbn/reporting-server");
var _rison = _interopRequireDefault(require("@kbn/rison"));
var _lib = require("../../../lib");
var _validator = require("./validator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const handleUnavailable = res => {
  return res.custom({
    statusCode: 503,
    body: 'Not Available'
  });
};
exports.handleUnavailable = handleUnavailable;
const ParamsValidation = _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.string());
const QueryValidation = _configSchema.schema.nullable(_configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())));
const BodyValidation = _configSchema.schema.nullable(_configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.any())));
/**
 * Handles the common parts of requests to generate or schedule a report
 * Serves report job handling in the context of the request to generate the report
 */
class RequestHandler {
  constructor(opts) {
    this.opts = opts;
  }
  static getValidation() {
    throw new Error('getValidation() must be implemented in a subclass');
  }
  getJobParams() {
    var _req$query;
    let jobParamsRison = null;
    const req = this.opts.req;
    const res = this.opts.res;
    if (req.body) {
      const {
        jobParams: jobParamsPayload
      } = req.body;
      jobParamsRison = jobParamsPayload ? jobParamsPayload : null;
    } else if ((_req$query = req.query) !== null && _req$query !== void 0 && _req$query.jobParams) {
      const {
        jobParams: queryJobParams
      } = req.query;
      if (queryJobParams) {
        jobParamsRison = queryJobParams;
      } else {
        jobParamsRison = null;
      }
    }
    if (!jobParamsRison) {
      throw res.customError({
        statusCode: 400,
        body: 'A jobParams RISON string is required in the querystring or POST body'
      });
    }
    let jobParams;
    try {
      jobParams = _rison.default.decode(jobParamsRison);
      if (!jobParams) {
        throw res.customError({
          statusCode: 400,
          body: 'Missing jobParams!'
        });
      }
    } catch (err) {
      throw res.customError({
        statusCode: 400,
        body: `invalid rison: ${jobParamsRison}`
      });
    }
    try {
      jobParams = (0, _validator.validateJobParams)(jobParams);
    } catch (err) {
      this.opts.logger.error(`Job param validation failed: ${err.message}`, {
        error: {
          stack_trace: err.stack
        }
      });
      throw res.customError({
        statusCode: 400,
        body: `invalid params: ${err.message}`
      });
    }
    return jobParams;
  }
  async createJob(exportTypeId, jobParams) {
    const exportType = this.opts.reporting.getExportTypesRegistry().getById(exportTypeId);
    if (exportType == null) {
      throw new Error(`Export type ${exportTypeId} does not exist in the registry!`);
    }
    if (!exportType.createJob) {
      throw new Error(`Export type ${exportTypeId} is not a valid instance!`);
    }

    // 1. Ensure the incoming params have a version field (should be set by the UI)
    const version = (0, _lib.checkParamsVersion)(jobParams, this.opts.logger);

    // 2. Create a payload object by calling exportType.createJob(), and adding some automatic parameters
    const job = await exportType.createJob(jobParams, this.opts.context, this.opts.req);
    return {
      job,
      version,
      jobType: exportType.jobType,
      name: exportType.name
    };
  }
  async checkLicense(exportTypeId) {
    const {
      reporting,
      context,
      res
    } = this.opts;

    // ensure the async dependencies are loaded
    if (!context.reporting) {
      return handleUnavailable(res);
    }
    const licenseInfo = await reporting.getLicenseInfo();
    const licenseResults = licenseInfo[exportTypeId];
    if (!licenseResults) {
      return res.badRequest({
        body: `Invalid export-type of ${exportTypeId}`
      });
    }
    if (!licenseResults.enableLinks) {
      return res.forbidden({
        body: licenseResults.message
      });
    }
    return null;
  }
  async encryptHeaders() {
    const {
      encryptionKey
    } = this.opts.reporting.getConfig();
    const crypto = (0, _reportingServer.cryptoFactory)(encryptionKey);
    return await crypto.encrypt(this.opts.req.headers);
  }
  handleError(err, counters, jobtype) {
    this.opts.logger.error(err);
    if (err instanceof _boom.default.Boom) {
      const statusCode = err.output.statusCode;
      counters === null || counters === void 0 ? void 0 : counters.errorCounter(jobtype, statusCode);
      return this.opts.res.customError({
        statusCode,
        body: err.output.payload.message
      });
    }
    counters === null || counters === void 0 ? void 0 : counters.errorCounter(jobtype, 500);
    return this.opts.res.customError({
      statusCode: 500,
      body: (err === null || err === void 0 ? void 0 : err.message) || _i18n.i18n.translate('xpack.reporting.errorHandler.unknownError', {
        defaultMessage: 'Unknown error'
      })
    });
  }
}
exports.RequestHandler = RequestHandler;