"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isExecutionError = isExecutionError;
exports.mapToReportingError = mapToReportingError;
var _common = require("@kbn/screenshotting-plugin/common");
var _reportingCommon = require("@kbn/reporting-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isExecutionError(error) {
  if (typeof error !== 'object' || error === null) {
    return false;
  }
  return ['name', 'message', 'stack'].every(k => k in error);
}
function getErrorName(error) {
  if (isExecutionError(error)) {
    return error.name;
  }
  return undefined;
}

/**
 * Map an error object from the Screenshotting plugin into an error type of the Reporting domain.
 *
 * NOTE: each type of ReportingError code must be referenced in each applicable `errorCodesSchema*` object in
 * x-pack/platform/plugins/private/reporting/server/usage/schema.ts
 *
 * @param {unknown} error - a kind of error object
 * @returns {ReportingError} - the converted error object
 */
function mapToReportingError(error) {
  if (error instanceof _reportingCommon.ReportingError) {
    return error;
  }
  const errorName = getErrorName(error);
  switch (true) {
    case error instanceof _common.errors.InvalidLayoutParametersError || errorName === 'InvalidLayoutParametersError':
      return new _reportingCommon.InvalidLayoutParametersError(error.message);
    case error instanceof _common.errors.DisallowedOutgoingUrl || errorName === 'DisallowedOutgoingUrl':
      return new _reportingCommon.DisallowedOutgoingUrl(error.message);
    case error instanceof _common.errors.BrowserClosedUnexpectedly || errorName === 'BrowserClosedUnexpectedly':
      return new _reportingCommon.BrowserUnexpectedlyClosedError(error.message);
    case error instanceof _common.errors.FailedToCaptureScreenshot || errorName === 'FailedToCaptureScreenshot':
      return new _reportingCommon.BrowserScreenshotError(error.message);
    case error instanceof _common.errors.FailedToSpawnBrowserError || errorName === 'FailedToSpawnBrowserError':
      return new _reportingCommon.BrowserCouldNotLaunchError();
    case error instanceof _common.errors.PdfWorkerOutOfMemoryError || errorName === 'PdfWorkerOutOfMemoryError':
      return new _reportingCommon.PdfWorkerOutOfMemoryError();
    case error instanceof _common.errors.InsufficientMemoryAvailableOnCloudError || errorName === 'InsufficientMemoryAvailableOnCloudError':
      return new _reportingCommon.VisualReportingSoftDisabledError();
  }
  return new _reportingCommon.UnknownError(error === null || error === void 0 ? void 0 : error.message);
}