"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FrameType = exports.FrameSymbolStatus = void 0;
exports.createStackFrameMetadata = createStackFrameMetadata;
exports.describeFrameType = describeFrameType;
exports.emptyStackTrace = exports.emptyStackFrame = exports.emptyExecutable = void 0;
exports.getCalleeFunction = getCalleeFunction;
exports.getCalleeLabel = getCalleeLabel;
exports.getCalleeSource = getCalleeSource;
exports.getFrameSymbolStatus = getFrameSymbolStatus;
exports.getLanguageType = getLanguageType;
exports.groupStackFrameMetadataByStackTrace = groupStackFrameMetadataByStackTrace;
exports.isErrorFrame = isErrorFrame;
exports.normalizeFrameType = normalizeFrameType;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/**
 * Stacktrace ID
 */
/**
 * StackFrame ID
 */
/**
 * File ID
 */
/**
 * Frame type
 * These frame types need to match with the constants defined in
 * https://github.com/open-telemetry/opentelemetry-ebpf-profiler/blob/main/libpf/frametype.go
 */
let FrameType = exports.FrameType = /*#__PURE__*/function (FrameType) {
  FrameType[FrameType["Unsymbolized"] = 0] = "Unsymbolized";
  FrameType[FrameType["Python"] = 1] = "Python";
  FrameType[FrameType["PHP"] = 2] = "PHP";
  FrameType[FrameType["Native"] = 3] = "Native";
  FrameType[FrameType["Kernel"] = 4] = "Kernel";
  FrameType[FrameType["JVM"] = 5] = "JVM";
  FrameType[FrameType["Ruby"] = 6] = "Ruby";
  FrameType[FrameType["Perl"] = 7] = "Perl";
  FrameType[FrameType["JavaScript"] = 8] = "JavaScript";
  FrameType[FrameType["PHPJIT"] = 9] = "PHPJIT";
  FrameType[FrameType["DotNET"] = 10] = "DotNET";
  FrameType[FrameType["Go"] = 11] = "Go";
  FrameType[FrameType["ErrorFlag"] = 128] = "ErrorFlag";
  FrameType[FrameType["Error"] = 255] = "Error";
  FrameType[FrameType["Root"] = 256] = "Root";
  FrameType[FrameType["ProcessName"] = 257] = "ProcessName";
  FrameType[FrameType["ThreadName"] = 258] = "ThreadName";
  FrameType[FrameType["ExecutableName"] = 259] = "ExecutableName";
  return FrameType;
}({});
const frameTypeDescriptions = {
  [FrameType.Unsymbolized]: '<unsymbolized frame>',
  [FrameType.Python]: 'Python',
  [FrameType.PHP]: 'PHP',
  [FrameType.Native]: 'Native',
  [FrameType.Kernel]: 'Kernel',
  [FrameType.JVM]: 'JVM/Hotspot',
  [FrameType.Ruby]: 'Ruby',
  [FrameType.Perl]: 'Perl',
  [FrameType.JavaScript]: 'JavaScript',
  [FrameType.PHPJIT]: 'PHP JIT',
  [FrameType.DotNET]: '.NET',
  [FrameType.Go]: 'Go',
  [FrameType.ErrorFlag]: 'ErrorFlag',
  [FrameType.Error]: 'Error',
  [FrameType.Root]: 'Root',
  [FrameType.ProcessName]: 'Process',
  [FrameType.ThreadName]: 'Thread',
  [FrameType.ExecutableName]: 'Executable'
};
function isErrorFrame(ft) {
  // eslint-disable-next-line no-bitwise
  return (ft & FrameType.ErrorFlag) !== 0;
}

/**
 * normalize the given frame type
 * @param ft FrameType
 * @returns FrameType
 */
function normalizeFrameType(ft) {
  // Normalize any frame type with error bit into our uniform error variant.
  if (isErrorFrame(ft)) {
    return FrameType.Error;
  }

  // Guard against new / unknown frame types, rewriting them to "unsymbolized".
  if (!(ft in frameTypeDescriptions)) {
    return FrameType.Unsymbolized;
  }
  return ft;
}

/**
 * get frame type name
 * @param ft FrameType
 * @returns string
 */
function describeFrameType(ft) {
  return frameTypeDescriptions[normalizeFrameType(ft)];
}

/** Stack trace */

/**
 * Empty stack trace
 */
const emptyStackTrace = exports.emptyStackTrace = {
  /** Frame IDs */
  FrameIDs: [],
  /** File IDs */
  FileIDs: [],
  /** Address or lines */
  AddressOrLines: [],
  /** Types */
  Types: [],
  selfAnnualCO2Kgs: 0,
  selfAnnualCostUSD: 0,
  Count: 0
};

/** Stack frame */

/**
 * Empty stack frame
 */
const emptyStackFrame = exports.emptyStackFrame = {
  /** File name */
  FileName: '',
  /** Function name */
  FunctionName: '',
  /** Function offset */
  FunctionOffset: 0,
  /** Line number */
  LineNumber: 0,
  /** Inline */
  Inline: false
};

/** Executable */

/**
 * Empty exectutable
 */
const emptyExecutable = exports.emptyExecutable = {
  /** file name */
  FileName: ''
};

/** Stack frame metadata */

/**
 * create stackframe metadata
 * @param options Partial<StackFrameMetadata>
 * @returns StackFrameMetadata
 */
function createStackFrameMetadata(options = {}) {
  var _options$FrameID, _options$FileID, _options$FrameType, _options$Inline, _options$AddressOrLin, _options$FunctionName, _options$FunctionOffs, _options$SourceFilena, _options$SourceLine, _options$ExeFileName;
  const metadata = {};
  metadata.FrameID = (_options$FrameID = options.FrameID) !== null && _options$FrameID !== void 0 ? _options$FrameID : '';
  metadata.FileID = (_options$FileID = options.FileID) !== null && _options$FileID !== void 0 ? _options$FileID : '';
  metadata.FrameType = (_options$FrameType = options.FrameType) !== null && _options$FrameType !== void 0 ? _options$FrameType : 0;
  metadata.Inline = (_options$Inline = options.Inline) !== null && _options$Inline !== void 0 ? _options$Inline : false;
  metadata.AddressOrLine = (_options$AddressOrLin = options.AddressOrLine) !== null && _options$AddressOrLin !== void 0 ? _options$AddressOrLin : 0;
  metadata.FunctionName = (_options$FunctionName = options.FunctionName) !== null && _options$FunctionName !== void 0 ? _options$FunctionName : '';
  metadata.FunctionOffset = (_options$FunctionOffs = options.FunctionOffset) !== null && _options$FunctionOffs !== void 0 ? _options$FunctionOffs : 0;
  metadata.SourceFilename = (_options$SourceFilena = options.SourceFilename) !== null && _options$SourceFilena !== void 0 ? _options$SourceFilena : '';
  metadata.SourceLine = (_options$SourceLine = options.SourceLine) !== null && _options$SourceLine !== void 0 ? _options$SourceLine : 0;
  metadata.ExeFileName = (_options$ExeFileName = options.ExeFileName) !== null && _options$ExeFileName !== void 0 ? _options$ExeFileName : '';
  return metadata;
}
function checkIfStringHasParentheses(s) {
  return /\(|\)/.test(s);
}
function getFunctionName(metadata) {
  return checkIfStringHasParentheses(metadata.FunctionName) ? metadata.FunctionName : `${metadata.FunctionName}()`;
}
function getExeFileName(metadata) {
  if ((metadata === null || metadata === void 0 ? void 0 : metadata.ExeFileName) === undefined) {
    return '';
  }
  if (metadata.ExeFileName !== '') {
    return metadata.ExeFileName;
  }
  return describeFrameType(metadata.FrameType);
}

/**
 * Get callee label
 * @param metadata StackFrameMetadata
 * @returns string
 */
function getCalleeLabel(metadata) {
  if (metadata.FrameType === FrameType.Error) {
    return `Error: unwinding error code #${metadata.AddressOrLine.toString()}`;
  }
  const inlineLabel = metadata.Inline ? '-> ' : '';
  if (metadata.FunctionName === '') {
    return `${inlineLabel}${getExeFileName(metadata)}`;
  }
  const sourceFilename = metadata.SourceFilename;
  const sourceURL = sourceFilename ? sourceFilename.split('/').pop() : '';
  if (!sourceURL) {
    return `${inlineLabel}${getExeFileName(metadata)}: ${getFunctionName(metadata)}`;
  }
  if (metadata.SourceLine === 0) {
    return `${inlineLabel}${getExeFileName(metadata)}: ${getFunctionName(metadata)} in ${sourceURL}`;
  }
  return `${inlineLabel}${getExeFileName(metadata)}: ${getFunctionName(metadata)} in ${sourceURL}#${metadata.SourceLine}`;
}
/**
 * Get callee function name
 * @param frame StackFrameMetadata
 * @returns string
 */
function getCalleeFunction(frame) {
  // In the best case scenario, we have the file names, source lines,
  // and function names. However we need to deal with missing function or
  // executable info.
  const exeDisplayName = frame.ExeFileName ? frame.ExeFileName : describeFrameType(frame.FrameType);

  // When there is no function name, only use the executable name
  return frame.FunctionName ? exeDisplayName + ': ' + frame.FunctionName : exeDisplayName;
}
/**
 * Frame symbol status
 */
let FrameSymbolStatus = exports.FrameSymbolStatus = /*#__PURE__*/function (FrameSymbolStatus) {
  FrameSymbolStatus["PARTIALLY_SYMBOLIZED"] = "PARTIALLY_SYMBOLIZED";
  FrameSymbolStatus["NOT_SYMBOLIZED"] = "NOT_SYMBOLIZED";
  FrameSymbolStatus["SYMBOLIZED"] = "SYMBOLIZED";
  return FrameSymbolStatus;
}({});
/** Frame symbols status params */
/**
 * Get frame symbol status
 * @param param FrameSymbolStatusParams
 * @returns FrameSymbolStatus
 */
function getFrameSymbolStatus(param) {
  const {
    sourceFilename,
    sourceLine,
    exeFileName
  } = param;
  if (sourceFilename === '' && sourceLine === 0) {
    if (exeFileName) {
      return FrameSymbolStatus.PARTIALLY_SYMBOLIZED;
    }
    return FrameSymbolStatus.NOT_SYMBOLIZED;
  }
  return FrameSymbolStatus.SYMBOLIZED;
}
const nativeLanguages = [FrameType.Native, FrameType.Kernel];
/**
 * Get language type
 * @param param LanguageTypeParams
 * @returns string
 */
function getLanguageType(param) {
  return nativeLanguages.includes(param.frameType) ? 'NATIVE' : 'INTERPRETED';
}

/**
 * Get callee source information.
 * If we don't have the executable filename, display <unsymbolized>
 * If no source line or filename available, display the executable offset
 * @param frame StackFrameMetadata
 * @returns string
 */
function getCalleeSource(frame) {
  if (frame.FrameType === FrameType.Error) {
    return `unwinding error code #${frame.AddressOrLine.toString()}`;
  }
  const frameSymbolStatus = getFrameSymbolStatus({
    sourceFilename: frame.SourceFilename,
    sourceLine: frame.SourceLine,
    exeFileName: frame.ExeFileName
  });
  switch (frameSymbolStatus) {
    case FrameSymbolStatus.NOT_SYMBOLIZED:
      {
        // If we don't have the executable filename, display <unsymbolized>
        return '<unsymbolized>';
      }
    case FrameSymbolStatus.PARTIALLY_SYMBOLIZED:
      {
        // If no source line or filename available, display the executable offset
        return frame.ExeFileName + (frame.AddressOrLine === 0 ? '' : '+0x' + frame.AddressOrLine.toString(16));
      }
    case FrameSymbolStatus.SYMBOLIZED:
      {
        return frame.SourceFilename + (frame.SourceLine !== 0 ? `#${frame.SourceLine}` : '');
      }
  }
}

/**
 * Group stackframe by stack trace
 * @param stackTraces Map<StackTraceID, StackTrace>
 * @param stackFrames Map<StackFrameID, StackFrame>
 * @param executables Map<FileID, Executable>
 * @returns Record<string, StackFrameMetadata[]>
 */
function groupStackFrameMetadataByStackTrace(stackTraces, stackFrames, executables) {
  const stackTraceMap = {};
  for (const [stackTraceID, trace] of stackTraces) {
    const numFramesPerTrace = trace.FrameIDs.length;
    const frameMetadata = new Array(numFramesPerTrace);
    for (let i = 0; i < numFramesPerTrace; i++) {
      var _stackFrames$get, _executables$get;
      const frameID = trace.FrameIDs[i];
      const fileID = trace.FileIDs[i];
      const addressOrLine = trace.AddressOrLines[i];
      const frame = (_stackFrames$get = stackFrames.get(frameID)) !== null && _stackFrames$get !== void 0 ? _stackFrames$get : emptyStackFrame;
      const executable = (_executables$get = executables.get(fileID)) !== null && _executables$get !== void 0 ? _executables$get : emptyExecutable;
      frameMetadata[i] = createStackFrameMetadata({
        FrameID: frameID,
        FileID: fileID,
        AddressOrLine: addressOrLine,
        FrameType: trace.Types[i],
        Inline: frame.Inline,
        FunctionName: frame.FunctionName,
        FunctionOffset: frame.FunctionOffset,
        SourceLine: frame.LineNumber,
        SourceFilename: frame.FileName,
        ExeFileName: executable.FileName
      });
    }
    stackTraceMap[stackTraceID] = frameMetadata;
  }
  return stackTraceMap;
}