"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DocumentationManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/task-manager-plugin/server");
var _inferenceCommon = require("@kbn/inference-common");
var _is_default_inference_endpoint = require("@kbn/product-doc-common/src/is_default_inference_endpoint");
var _tasks = require("../../tasks");
var _check_license = require("./check_license");
var _install_all = require("../../tasks/install_all");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TEN_MIN_IN_MS = 10 * 60 * 1000;

/**
 * High-level installation service, handling product documentation
 * installation as unary operations, abstracting away the fact
 * that documentation is composed of multiple entities.
 */
class DocumentationManager {
  constructor({
    logger,
    taskManager,
    licensing,
    docInstallClient,
    auditService,
    packageInstaller
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "taskManager", void 0);
    (0, _defineProperty2.default)(this, "licensing", void 0);
    (0, _defineProperty2.default)(this, "docInstallClient", void 0);
    (0, _defineProperty2.default)(this, "auditService", void 0);
    (0, _defineProperty2.default)(this, "packageInstaller", void 0);
    this.logger = logger;
    this.taskManager = taskManager;
    this.licensing = licensing;
    this.docInstallClient = docInstallClient;
    this.auditService = auditService;
    this.packageInstaller = packageInstaller;
  }
  async install(options) {
    var _options$inferenceId;
    const {
      request,
      force = false,
      wait = false
    } = options;
    const inferenceId = (_options$inferenceId = options.inferenceId) !== null && _options$inferenceId !== void 0 ? _options$inferenceId : _inferenceCommon.defaultInferenceEndpoints.ELSER;
    const {
      status
    } = await this.getStatus({
      inferenceId
    });
    if (!force && status === 'installed') {
      return;
    }
    const license = await this.licensing.getLicense();
    if (!(0, _check_license.checkLicense)(license)) {
      throw new Error('Elastic documentation requires an enterprise license');
    }
    const taskId = await (0, _tasks.scheduleInstallAllTask)({
      taskManager: this.taskManager,
      logger: this.logger,
      inferenceId
    });
    if (request) {
      this.auditService.asScoped(request).log({
        message: `User is requesting installation of product documentation for AI Assistants. Task ID=[${taskId}]` + (inferenceId ? `| Inference ID=[${inferenceId}]` : ''),
        event: {
          action: 'product_documentation_create',
          category: ['database'],
          type: ['creation'],
          outcome: 'unknown'
        }
      });
    }
    if (wait) {
      await (0, _tasks.waitUntilTaskCompleted)({
        taskManager: this.taskManager,
        taskId,
        timeout: TEN_MIN_IN_MS
      });
    }
  }
  async update(options) {
    const {
      request,
      wait = false,
      inferenceId,
      forceUpdate
    } = options;
    const taskId = await (0, _tasks.scheduleEnsureUpToDateTask)({
      taskManager: this.taskManager,
      logger: this.logger,
      inferenceId,
      forceUpdate
    });
    if (request) {
      this.auditService.asScoped(request).log({
        message: `User is requesting update of product documentation for AI Assistants. Task ID=[${taskId}]` + (inferenceId ? `| Inference ID=[${inferenceId}]` : ''),
        event: {
          action: 'product_documentation_update',
          category: ['database'],
          type: ['change'],
          outcome: 'unknown'
        }
      });
    }
    if (wait) {
      await (0, _tasks.waitUntilTaskCompleted)({
        taskManager: this.taskManager,
        taskId,
        timeout: TEN_MIN_IN_MS
      });
    }
  }
  async updateAll(options) {
    var _await$this$docInstal;
    const {
      forceUpdate,
      inferenceIds
    } = options !== null && options !== void 0 ? options : {};
    const idsToUpdate = Array.isArray(inferenceIds) && (inferenceIds === null || inferenceIds === void 0 ? void 0 : inferenceIds.length) > 0 ? inferenceIds : (_await$this$docInstal = await this.docInstallClient.getPreviouslyInstalledInferenceIds()) !== null && _await$this$docInstal !== void 0 ? _await$this$docInstal : [];
    this.logger.info(`Updating product documentation to latest version for Inference IDs: ${idsToUpdate}`);
    await Promise.all(idsToUpdate.map(inferenceId => this.update({
      inferenceId,
      forceUpdate
    })));
    return {
      inferenceIds: idsToUpdate
    };
  }
  async updateSecurityLabsAll(options) {
    var _await$this$docInstal2;
    const {
      forceUpdate
    } = options !== null && options !== void 0 ? options : {};
    const idsToUpdate = (_await$this$docInstal2 = await this.docInstallClient.getPreviouslyInstalledSecurityLabsInferenceIds()) !== null && _await$this$docInstal2 !== void 0 ? _await$this$docInstal2 : [];
    if (idsToUpdate.length === 0) {
      return {
        inferenceIds: []
      };
    }
    this.logger.info(`Updating Security Labs content to latest version for Inference IDs: ${idsToUpdate}`);
    await Promise.all(idsToUpdate.map(inferenceId => (0, _tasks.scheduleEnsureSecurityLabsUpToDateTask)({
      taskManager: this.taskManager,
      logger: this.logger,
      inferenceId,
      forceUpdate
    })));
    return {
      inferenceIds: idsToUpdate
    };
  }
  async uninstall(options) {
    const {
      request,
      wait = false,
      inferenceId
    } = options;
    const taskId = await (0, _tasks.scheduleUninstallAllTask)({
      taskManager: this.taskManager,
      logger: this.logger,
      inferenceId
    });
    if (request) {
      this.auditService.asScoped(request).log({
        message: `User is requesting deletion of product documentation for AI Assistants. Task ID=[${taskId}]`,
        event: {
          action: 'product_documentation_delete',
          category: ['database'],
          type: ['deletion'],
          outcome: 'unknown'
        }
      });
    }
    if (wait) {
      await (0, _tasks.waitUntilTaskCompleted)({
        taskManager: this.taskManager,
        taskId,
        timeout: TEN_MIN_IN_MS
      });
    }
  }

  /**
   * @param inferenceId - The inference ID to get the status for. If not provided, the default ELSER inference ID will be used.
   */
  async getStatus({
    inferenceId
  }) {
    const taskId = (0, _is_default_inference_endpoint.isImpliedDefaultElserInferenceId)(inferenceId) ? _tasks.INSTALL_ALL_TASK_ID : _install_all.INSTALL_ALL_TASK_ID_MULTILINGUAL;
    const taskStatus = await (0, _tasks.getTaskStatus)({
      taskManager: this.taskManager,
      taskId
    });
    if (taskStatus !== 'not_scheduled') {
      const status = convertTaskStatus(taskStatus);
      if (status !== 'unknown') {
        return {
          status
        };
      }
    }
    const installStatus = await this.docInstallClient.getInstallationStatus({
      inferenceId
    });
    const overallStatus = getOverallStatus(Object.values(installStatus).map(v => v.status));
    return {
      status: overallStatus,
      installStatus
    };
  }
  async getStatuses({
    inferenceIds
  }) {
    // check status after installation in case of failure
    const statuses = await Promise.allSettled(inferenceIds.map(inferenceId => this.getStatus({
      inferenceId
    })));
    const body = statuses.reduce((acc, installationStatus, index) => {
      const inferenceId = inferenceIds[index];
      // Handle internal server error
      if (installationStatus.status === 'rejected') {
        const failureReason = installationStatus.reason;
        return {
          ...acc,
          [inferenceId]: {
            installed: status === 'uninstalled',
            ...(failureReason ? {
              failureReason: JSON.stringify(failureReason)
            } : {})
          }
        };
      }
      if (installationStatus.status === 'fulfilled') {
        const {
          status,
          installStatus
        } = installationStatus.value;
        let failureReason = null;
        // Check for real reason of previous installation failure
        if (status === 'error' && installStatus) {
          failureReason = Object.values(installStatus).filter(product => product.status === 'error' && product.failureReason).map(product => product.failureReason).join('\n');
        }
        return {
          ...acc,
          [inferenceId]: {
            installed: status === 'installed',
            ...(failureReason ? {
              failureReason
            } : {})
          }
        };
      }
      return acc;
    }, {});
    return body;
  }

  // Security Labs methods

  async installSecurityLabs(options) {
    const {
      request,
      inferenceId,
      version
    } = options;
    const license = await this.licensing.getLicense();
    if (!(0, _check_license.checkLicense)(license)) {
      throw new Error('Security Labs content requires an enterprise license');
    }
    if (!this.packageInstaller) {
      throw new Error('PackageInstaller not available');
    }
    if (request) {
      this.auditService.asScoped(request).log({
        message: `User is requesting installation of Security Labs content for AI Assistants.` + (inferenceId ? ` Inference ID=[${inferenceId}]` : '') + (version ? ` Version=[${version}]` : ''),
        event: {
          action: 'security_labs_create',
          category: ['database'],
          type: ['creation'],
          outcome: 'unknown'
        }
      });
    }
    try {
      await this.packageInstaller.installSecurityLabs({
        version,
        inferenceId
      });
    } catch (error) {
      this.logger.error(`Failed to install Security Labs content: ${error.message}`);
      throw error;
    }
  }
  async uninstallSecurityLabs(options) {
    const {
      request,
      inferenceId
    } = options;
    if (!this.packageInstaller) {
      throw new Error('PackageInstaller not available');
    }
    if (request) {
      this.auditService.asScoped(request).log({
        message: `User is requesting deletion of Security Labs content for AI Assistants.`,
        event: {
          action: 'security_labs_delete',
          category: ['database'],
          type: ['deletion'],
          outcome: 'unknown'
        }
      });
    }
    try {
      await this.packageInstaller.uninstallSecurityLabs({
        inferenceId
      });
    } catch (error) {
      this.logger.error(`Failed to uninstall Security Labs content: ${error.message}`);
      throw error;
    }
  }
  async getSecurityLabsStatus({
    inferenceId
  }) {
    if (!this.packageInstaller) {
      return {
        status: 'uninstalled'
      };
    }
    try {
      return await this.packageInstaller.getSecurityLabsStatus({
        inferenceId
      });
    } catch (error) {
      this.logger.error(`Failed to get Security Labs status: ${error.message}`);
      return {
        status: 'error',
        failureReason: error.message
      };
    }
  }
}
exports.DocumentationManager = DocumentationManager;
const convertTaskStatus = taskStatus => {
  switch (taskStatus) {
    case _server.TaskStatus.Idle:
    case _server.TaskStatus.Claiming:
    case _server.TaskStatus.Running:
      return 'installing';
    case _server.TaskStatus.Failed:
      return 'error';
    case _server.TaskStatus.Unrecognized:
    case _server.TaskStatus.DeadLetter:
    case _server.TaskStatus.ShouldDelete:
    default:
      return 'unknown';
  }
};
const getOverallStatus = statuses => {
  const statusOrder = ['error', 'installing', 'uninstalling', 'uninstalled', 'installed'];
  for (const status of statusOrder) {
    if (statuses.includes(status)) {
      return status;
    }
  }
  return 'installed';
};