"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isIndexNotFoundError = exports.QueryUtils = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _onechatServer = require("@kbn/onechat-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isIndexNotFoundError = error => {
  var _castError$attributes, _castError$attributes2, _castError$attributes3, _castError$attributes4, _castError$attributes5, _castError$meta, _castError$meta$body, _castError$meta$body$, _castError$meta2, _castError$meta2$body, _castError$meta2$body2, _castError$meta2$body3;
  if (!error || typeof error !== 'object') {
    return false;
  }
  const castError = error;

  // Check various error structure formats from Elasticsearch client
  return ((_castError$attributes = castError.attributes) === null || _castError$attributes === void 0 ? void 0 : (_castError$attributes2 = _castError$attributes.caused_by) === null || _castError$attributes2 === void 0 ? void 0 : _castError$attributes2.type) === 'index_not_found_exception' || ((_castError$attributes3 = castError.attributes) === null || _castError$attributes3 === void 0 ? void 0 : (_castError$attributes4 = _castError$attributes3.error) === null || _castError$attributes4 === void 0 ? void 0 : (_castError$attributes5 = _castError$attributes4.caused_by) === null || _castError$attributes5 === void 0 ? void 0 : _castError$attributes5.type) === 'index_not_found_exception' || ((_castError$meta = castError.meta) === null || _castError$meta === void 0 ? void 0 : (_castError$meta$body = _castError$meta.body) === null || _castError$meta$body === void 0 ? void 0 : (_castError$meta$body$ = _castError$meta$body.error) === null || _castError$meta$body$ === void 0 ? void 0 : _castError$meta$body$.type) === 'index_not_found_exception' || ((_castError$meta2 = castError.meta) === null || _castError$meta2 === void 0 ? void 0 : (_castError$meta2$body = _castError$meta2.body) === null || _castError$meta2$body === void 0 ? void 0 : (_castError$meta2$body2 = _castError$meta2$body.error) === null || _castError$meta2$body2 === void 0 ? void 0 : (_castError$meta2$body3 = _castError$meta2$body2.caused_by) === null || _castError$meta2$body3 === void 0 ? void 0 : _castError$meta2$body3.type) === 'index_not_found_exception' || typeof castError.message === 'string' && castError.message.includes('index_not_found_exception');
};

/**
 * Usage counter data from saved objects
 */
exports.isIndexNotFoundError = isIndexNotFoundError;
/**
 * Query utilities for telemetry data collection
 *
 * Provides helpers for querying:
 * - Usage counters from saved objects
 * - Conversation data from Elasticsearch
 * - Custom tools and agents from Elasticsearch
 */
class QueryUtils {
  constructor(esClient, soClient, logger) {
    /**
     * TTFT/TTLT percentile metrics structure
     */
    (0, _defineProperty2.default)(this, "defaultTimingMetrics", {
      p50: 0,
      p75: 0,
      p90: 0,
      p95: 0,
      p99: 0,
      mean: 0,
      total_samples: 0
    });
    this.esClient = esClient;
    this.soClient = soClient;
    this.logger = logger;
  }

  /**
   * Get all usage counters for a specific domain
   * @param domainId - Domain identifier (e.g., 'onechat')
   * @returns Array of usage counter data
   */
  async getCountersByDomain(domainId) {
    try {
      const {
        saved_objects: savedObjects
      } = await this.soClient.find({
        type: 'usage-counter',
        perPage: 10000,
        filter: `usage-counter.attributes.domainId:"${domainId}"`
      });
      return savedObjects.map(so => ({
        counterName: so.attributes.counterName,
        counterType: so.attributes.counterType,
        count: so.attributes.count,
        domainId: so.attributes.domainId
      }));
    } catch (error) {
      this.logger.error(`Failed to query usage counters: ${error.message}`);
      return [];
    }
  }

  /**
   * Get usage counters filtered by name prefix
   * @param domainId - Domain identifier
   * @param prefix - Counter name prefix (e.g., 'tool_call_')
   * @returns Map of counter name → count
   */
  async getCountersByPrefix(domainId, prefix) {
    try {
      const allCounters = await this.getCountersByDomain(domainId);
      const filtered = allCounters.filter(counter => counter.counterName.startsWith(prefix));
      const result = new Map();
      for (const counter of filtered) {
        result.set(counter.counterName, counter.count);
      }
      return result;
    } catch (error) {
      this.logger.error(`Failed to query counters by prefix "${prefix}": ${error.message}`);
      return new Map();
    }
  }
  async getCustomToolsMetrics() {
    try {
      var _response$aggregation, _response$aggregation2;
      const toolIndexName = (0, _onechatServer.chatSystemIndex)('tools');
      const response = await this.esClient.search({
        index: toolIndexName,
        size: 0,
        aggs: {
          by_type: {
            terms: {
              field: 'type',
              size: 100
            }
          }
        },
        query: {
          bool: {
            must_not: [{
              term: {
                type: 'builtin'
              }
            }]
          }
        }
      });
      const buckets = ((_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : (_response$aggregation2 = _response$aggregation.by_type) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.buckets) || [];
      const byType = buckets.map(bucket => ({
        type: bucket.key,
        count: bucket.doc_count
      }));
      const total = buckets.reduce((sum, bucket) => sum + bucket.doc_count, 0);
      return {
        total,
        by_type: byType
      };
    } catch (error) {
      // Suppress warning for missing index - expected when no tools have been created yet
      if (!isIndexNotFoundError(error)) {
        this.logger.warn(`Failed to fetch custom tools counts: ${error.message}`);
      }
      return {
        total: 0,
        by_type: []
      };
    }
  }

  /**
   * Get counts of custom agents from Elasticsearch
   */
  async getCustomAgentsMetrics() {
    try {
      const agentsIndexName = (0, _onechatServer.chatSystemIndex)('agents');
      const response = await this.esClient.count({
        index: agentsIndexName
      });
      return response.count || 0;
    } catch (error) {
      // Suppress warning for missing index - expected when no agents have been created yet
      if (!isIndexNotFoundError(error)) {
        this.logger.warn(`Failed to fetch custom agents count: ${error.message}`);
      }
      return 0;
    }
  }

  /**
   * Get conversation metrics from Elasticsearch
   */
  async getConversationMetrics() {
    try {
      var _response$aggregation3, _response$aggregation4, _response$aggregation5;
      const conversationIndexName = (0, _onechatServer.chatSystemIndex)('conversations');
      const response = await this.esClient.search({
        index: conversationIndexName,
        size: 0,
        aggs: {
          rounds_distribution: {
            terms: {
              script: {
                source: `
                def source = params._source;
                def roundsArray = source.conversation_rounds != null ? source.conversation_rounds : source.rounds;
                def rounds = roundsArray != null ? roundsArray.size() : 0;

                if (rounds <= 5) return '1-5';
                if (rounds <= 10) return '6-10';
                if (rounds <= 20) return '11-20';
                if (rounds <= 50) return '21-50';
                return '51+';
              `,
                lang: 'painless'
              },
              size: 100
            }
          },
          total_tokens: {
            sum: {
              script: {
                source: `
                  def source = params._source;
                  def roundsArray = source.conversation_rounds != null ? source.conversation_rounds : source.rounds;
                  def totalTokens = 0;
                  if (roundsArray != null) {
                    for (def round : roundsArray) {
                      if (round.model_usage != null) {
                        def inputTokens = round.model_usage.input_tokens != null ? round.model_usage.input_tokens : 0;
                        def outputTokens = round.model_usage.output_tokens != null ? round.model_usage.output_tokens : 0;
                        totalTokens += inputTokens + outputTokens;
                      }
                    }
                  }
                  return totalTokens;
                `,
                lang: 'painless'
              }
            }
          }
        }
      });
      const buckets = ((_response$aggregation3 = response.aggregations) === null || _response$aggregation3 === void 0 ? void 0 : (_response$aggregation4 = _response$aggregation3.rounds_distribution) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.buckets) || [];
      const roundsDistribution = buckets.map(bucket => ({
        bucket: bucket.key,
        count: bucket.doc_count
      }));

      // Calculate total rounds and average
      let totalRounds = 0;
      buckets.forEach(bucket => {
        const bucketKey = bucket.key;
        const count = bucket.doc_count;
        if (bucketKey === '1-5') {
          totalRounds += count * 3; // Approximate: use middle value
        } else if (bucketKey === '6-10') {
          totalRounds += count * 8;
        } else if (bucketKey === '11-20') {
          totalRounds += count * 15;
        } else if (bucketKey === '21-50') {
          totalRounds += count * 35;
        } else {
          totalRounds += count * 75; // Approximate for 51+
        }
      });
      const total = response.hits.total;
      const totalConversations = typeof total === 'number' ? total : (total === null || total === void 0 ? void 0 : total.value) || 0;
      const avgRoundsPerConversation = totalConversations > 0 ? totalRounds / totalConversations : 0;
      const totalTokensAgg = (_response$aggregation5 = response.aggregations) === null || _response$aggregation5 === void 0 ? void 0 : _response$aggregation5.total_tokens;
      const tokensUsed = (totalTokensAgg === null || totalTokensAgg === void 0 ? void 0 : totalTokensAgg.value) || 0;
      const averageTokensPerConversation = totalConversations > 0 ? tokensUsed / totalConversations : 0;
      return {
        total: totalConversations,
        total_rounds: totalRounds,
        avg_rounds_per_conversation: Math.round(avgRoundsPerConversation * 100) / 100,
        rounds_distribution: roundsDistribution,
        tokens_used: Math.round(tokensUsed),
        average_tokens_per_conversation: Math.round(averageTokensPerConversation * 100) / 100
      };
    } catch (error) {
      if (!isIndexNotFoundError(error)) {
        this.logger.warn(`Failed to fetch conversation metrics: ${error.message}`);
      }
      return {
        total: 0,
        total_rounds: 0,
        avg_rounds_per_conversation: 0,
        rounds_distribution: [],
        tokens_used: 0,
        average_tokens_per_conversation: 0
      };
    }
  }
  /**
   * Get Time-to-First-Token (TTFT) metrics from conversation rounds
   * Queries the conversations index and aggregates TTFT data
   */
  async getTTFTMetrics() {
    try {
      var _response$aggregation6, _aggs$ttft_percentile, _aggs$ttft_avg, _aggs$ttft_count;
      const conversationIndexName = (0, _onechatServer.chatSystemIndex)('conversations');
      const response = await this.esClient.search({
        index: conversationIndexName,
        size: 0,
        aggs: {
          all_rounds: {
            nested: {
              path: 'conversation_rounds'
            },
            aggs: {
              ttft_percentiles: {
                percentiles: {
                  field: 'conversation_rounds.time_to_first_token',
                  percents: [50, 75, 90, 95, 99]
                }
              },
              ttft_avg: {
                avg: {
                  field: 'conversation_rounds.time_to_first_token'
                }
              },
              ttft_count: {
                value_count: {
                  field: 'conversation_rounds.time_to_first_token'
                }
              }
            }
          }
        }
      });
      const aggs = (_response$aggregation6 = response.aggregations) === null || _response$aggregation6 === void 0 ? void 0 : _response$aggregation6.all_rounds;
      if (!aggs) {
        return {
          ...this.defaultTimingMetrics
        };
      }
      const percentiles = ((_aggs$ttft_percentile = aggs.ttft_percentiles) === null || _aggs$ttft_percentile === void 0 ? void 0 : _aggs$ttft_percentile.values) || {};
      return {
        p50: Math.round(percentiles['50.0'] || 0),
        p75: Math.round(percentiles['75.0'] || 0),
        p90: Math.round(percentiles['90.0'] || 0),
        p95: Math.round(percentiles['95.0'] || 0),
        p99: Math.round(percentiles['99.0'] || 0),
        mean: Math.round(((_aggs$ttft_avg = aggs.ttft_avg) === null || _aggs$ttft_avg === void 0 ? void 0 : _aggs$ttft_avg.value) || 0),
        total_samples: ((_aggs$ttft_count = aggs.ttft_count) === null || _aggs$ttft_count === void 0 ? void 0 : _aggs$ttft_count.value) || 0
      };
    } catch (error) {
      if (!isIndexNotFoundError(error)) {
        this.logger.warn(`Failed to fetch TTFT metrics: ${error.message}`);
      }
      return {
        ...this.defaultTimingMetrics
      };
    }
  }

  /**
   * Get Time-to-Last-Token (TTLT) metrics from conversation rounds
   * Queries the conversations index and aggregates TTLT data
   */
  async getTTLTMetrics() {
    try {
      var _response$aggregation7, _aggs$ttlt_percentile, _aggs$ttlt_avg, _aggs$ttlt_count;
      const conversationIndexName = (0, _onechatServer.chatSystemIndex)('conversations');
      const response = await this.esClient.search({
        index: conversationIndexName,
        size: 0,
        aggs: {
          all_rounds: {
            nested: {
              path: 'conversation_rounds'
            },
            aggs: {
              ttlt_percentiles: {
                percentiles: {
                  field: 'conversation_rounds.time_to_last_token',
                  percents: [50, 75, 90, 95, 99]
                }
              },
              ttlt_avg: {
                avg: {
                  field: 'conversation_rounds.time_to_last_token'
                }
              },
              ttlt_count: {
                value_count: {
                  field: 'conversation_rounds.time_to_last_token'
                }
              }
            }
          }
        }
      });
      const aggs = (_response$aggregation7 = response.aggregations) === null || _response$aggregation7 === void 0 ? void 0 : _response$aggregation7.all_rounds;
      if (!aggs) {
        return {
          ...this.defaultTimingMetrics
        };
      }
      const percentiles = ((_aggs$ttlt_percentile = aggs.ttlt_percentiles) === null || _aggs$ttlt_percentile === void 0 ? void 0 : _aggs$ttlt_percentile.values) || {};
      return {
        p50: Math.round(percentiles['50.0'] || 0),
        p75: Math.round(percentiles['75.0'] || 0),
        p90: Math.round(percentiles['90.0'] || 0),
        p95: Math.round(percentiles['95.0'] || 0),
        p99: Math.round(percentiles['99.0'] || 0),
        mean: Math.round(((_aggs$ttlt_avg = aggs.ttlt_avg) === null || _aggs$ttlt_avg === void 0 ? void 0 : _aggs$ttlt_avg.value) || 0),
        total_samples: ((_aggs$ttlt_count = aggs.ttlt_count) === null || _aggs$ttlt_count === void 0 ? void 0 : _aggs$ttlt_count.value) || 0
      };
    } catch (error) {
      if (!isIndexNotFoundError(error)) {
        this.logger.warn(`Failed to fetch TTLT metrics: ${error.message}`);
      }
      return {
        ...this.defaultTimingMetrics
      };
    }
  }

  /**
   * Get latency breakdown by model
   * Returns TTFT and TTLT p50/p95 for each model
   * Uses stored model info from conversation rounds
   */
  async getLatencyByModel() {
    try {
      var _response$aggregation8, _aggs$by_model;
      const conversationIndexName = (0, _onechatServer.chatSystemIndex)('conversations');
      const response = await this.esClient.search({
        index: conversationIndexName,
        size: 0,
        aggs: {
          all_rounds: {
            nested: {
              path: 'conversation_rounds'
            },
            aggs: {
              by_model: {
                terms: {
                  field: 'conversation_rounds.model_usage.model',
                  size: 50
                },
                aggs: {
                  ttft_percentiles: {
                    percentiles: {
                      field: 'conversation_rounds.time_to_first_token',
                      percents: [50, 95]
                    }
                  },
                  ttlt_percentiles: {
                    percentiles: {
                      field: 'conversation_rounds.time_to_last_token',
                      percents: [50, 95]
                    }
                  }
                }
              },
              by_connector: {
                terms: {
                  field: 'conversation_rounds.model_usage.connector_id',
                  size: 50
                },
                aggs: {
                  ttft_percentiles: {
                    percentiles: {
                      field: 'conversation_rounds.time_to_first_token',
                      percents: [50, 95]
                    }
                  },
                  ttlt_percentiles: {
                    percentiles: {
                      field: 'conversation_rounds.time_to_last_token',
                      percents: [50, 95]
                    }
                  }
                }
              }
            }
          }
        }
      });
      const aggs = (_response$aggregation8 = response.aggregations) === null || _response$aggregation8 === void 0 ? void 0 : _response$aggregation8.all_rounds;
      const modelBuckets = (aggs === null || aggs === void 0 ? void 0 : (_aggs$by_model = aggs.by_model) === null || _aggs$by_model === void 0 ? void 0 : _aggs$by_model.buckets) || [];
      if (modelBuckets.length === 0) {
        return [];
      }
      const results = [];

      // Process rounds grouped by stored model field
      for (const bucket of modelBuckets) {
        var _bucket$ttft_percenti, _bucket$ttft_percenti2, _bucket$ttft_percenti3, _bucket$ttft_percenti4, _bucket$ttlt_percenti, _bucket$ttlt_percenti2, _bucket$ttlt_percenti3, _bucket$ttlt_percenti4;
        const model = bucket.key;
        const ttftp50 = ((_bucket$ttft_percenti = bucket.ttft_percentiles) === null || _bucket$ttft_percenti === void 0 ? void 0 : (_bucket$ttft_percenti2 = _bucket$ttft_percenti.values) === null || _bucket$ttft_percenti2 === void 0 ? void 0 : _bucket$ttft_percenti2['50.0']) || 0;
        const ttftp95 = ((_bucket$ttft_percenti3 = bucket.ttft_percentiles) === null || _bucket$ttft_percenti3 === void 0 ? void 0 : (_bucket$ttft_percenti4 = _bucket$ttft_percenti3.values) === null || _bucket$ttft_percenti4 === void 0 ? void 0 : _bucket$ttft_percenti4['95.0']) || 0;
        const ttltp50 = ((_bucket$ttlt_percenti = bucket.ttlt_percentiles) === null || _bucket$ttlt_percenti === void 0 ? void 0 : (_bucket$ttlt_percenti2 = _bucket$ttlt_percenti.values) === null || _bucket$ttlt_percenti2 === void 0 ? void 0 : _bucket$ttlt_percenti2['50.0']) || 0;
        const ttltp95 = ((_bucket$ttlt_percenti3 = bucket.ttlt_percentiles) === null || _bucket$ttlt_percenti3 === void 0 ? void 0 : (_bucket$ttlt_percenti4 = _bucket$ttlt_percenti3.values) === null || _bucket$ttlt_percenti4 === void 0 ? void 0 : _bucket$ttlt_percenti4['95.0']) || 0;
        const sampleCount = bucket.doc_count || 0;
        if (sampleCount === 0) continue;
        results.push({
          model,
          ttft_p50: Math.round(ttftp50),
          ttft_p95: Math.round(ttftp95),
          ttlt_p50: Math.round(ttltp50),
          ttlt_p95: Math.round(ttltp95),
          sample_count: sampleCount
        });
      }

      // Sort by sample count descending
      results.sort((a, b) => b.sample_count - a.sample_count);
      return results;
    } catch (error) {
      if (!isIndexNotFoundError(error)) {
        this.logger.warn(`Failed to fetch latency by model: ${error.message}`);
      }
      return [];
    }
  }

  /**
   * Get latency breakdown by agent
   * Returns TTFT and TTLT p50/p95 for each agent_id
   */
  async getLatencyByAgentType() {
    try {
      var _response$aggregation9, _response$aggregation10;
      const conversationIndexName = (0, _onechatServer.chatSystemIndex)('conversations');

      // Get latency metrics grouped by agent_id
      const response = await this.esClient.search({
        index: conversationIndexName,
        size: 0,
        aggs: {
          by_agent: {
            terms: {
              field: 'agent_id',
              size: 50
            },
            aggs: {
              all_rounds: {
                nested: {
                  path: 'conversation_rounds'
                },
                aggs: {
                  ttft_percentiles: {
                    percentiles: {
                      field: 'conversation_rounds.time_to_first_token',
                      percents: [50, 95]
                    }
                  },
                  ttlt_percentiles: {
                    percentiles: {
                      field: 'conversation_rounds.time_to_last_token',
                      percents: [50, 95]
                    }
                  },
                  count: {
                    value_count: {
                      field: 'conversation_rounds.time_to_first_token'
                    }
                  }
                }
              }
            }
          }
        }
      });
      const buckets = ((_response$aggregation9 = response.aggregations) === null || _response$aggregation9 === void 0 ? void 0 : (_response$aggregation10 = _response$aggregation9.by_agent) === null || _response$aggregation10 === void 0 ? void 0 : _response$aggregation10.buckets) || [];
      return buckets.map(bucket => {
        var _roundsAggs$ttft_perc, _roundsAggs$ttft_perc2, _roundsAggs$ttft_perc3, _roundsAggs$ttft_perc4, _roundsAggs$ttlt_perc, _roundsAggs$ttlt_perc2, _roundsAggs$ttlt_perc3, _roundsAggs$ttlt_perc4, _roundsAggs$count;
        const roundsAggs = bucket.all_rounds;
        return {
          agent_id: bucket.key,
          ttft_p50: Math.round((roundsAggs === null || roundsAggs === void 0 ? void 0 : (_roundsAggs$ttft_perc = roundsAggs.ttft_percentiles) === null || _roundsAggs$ttft_perc === void 0 ? void 0 : (_roundsAggs$ttft_perc2 = _roundsAggs$ttft_perc.values) === null || _roundsAggs$ttft_perc2 === void 0 ? void 0 : _roundsAggs$ttft_perc2['50.0']) || 0),
          ttft_p95: Math.round((roundsAggs === null || roundsAggs === void 0 ? void 0 : (_roundsAggs$ttft_perc3 = roundsAggs.ttft_percentiles) === null || _roundsAggs$ttft_perc3 === void 0 ? void 0 : (_roundsAggs$ttft_perc4 = _roundsAggs$ttft_perc3.values) === null || _roundsAggs$ttft_perc4 === void 0 ? void 0 : _roundsAggs$ttft_perc4['95.0']) || 0),
          ttlt_p50: Math.round((roundsAggs === null || roundsAggs === void 0 ? void 0 : (_roundsAggs$ttlt_perc = roundsAggs.ttlt_percentiles) === null || _roundsAggs$ttlt_perc === void 0 ? void 0 : (_roundsAggs$ttlt_perc2 = _roundsAggs$ttlt_perc.values) === null || _roundsAggs$ttlt_perc2 === void 0 ? void 0 : _roundsAggs$ttlt_perc2['50.0']) || 0),
          ttlt_p95: Math.round((roundsAggs === null || roundsAggs === void 0 ? void 0 : (_roundsAggs$ttlt_perc3 = roundsAggs.ttlt_percentiles) === null || _roundsAggs$ttlt_perc3 === void 0 ? void 0 : (_roundsAggs$ttlt_perc4 = _roundsAggs$ttlt_perc3.values) === null || _roundsAggs$ttlt_perc4 === void 0 ? void 0 : _roundsAggs$ttlt_perc4['95.0']) || 0),
          sample_count: (roundsAggs === null || roundsAggs === void 0 ? void 0 : (_roundsAggs$count = roundsAggs.count) === null || _roundsAggs$count === void 0 ? void 0 : _roundsAggs$count.value) || 0
        };
      });
    } catch (error) {
      if (!isIndexNotFoundError(error)) {
        this.logger.warn(`Failed to fetch latency by agent: ${error.message}`);
      }
      return [];
    }
  }

  /**
   * Calculate percentiles from bucketed time data
   * @param buckets - Map of bucket name → count
   * @returns Calculated percentiles (p50, p75, p90, p95, p99, mean)
   */
  calculatePercentilesFromBuckets(buckets, domainPrefix = 'onechat') {
    // Bucket boundaries (in milliseconds) - keys include domain prefix
    const bucketRanges = {
      [`${domainPrefix}_query_to_result_time_<1s`]: {
        min: 0,
        max: 1000,
        mid: 500
      },
      [`${domainPrefix}_query_to_result_time_1-5s`]: {
        min: 1000,
        max: 5000,
        mid: 3000
      },
      [`${domainPrefix}_query_to_result_time_5-10s`]: {
        min: 5000,
        max: 10000,
        mid: 7500
      },
      [`${domainPrefix}_query_to_result_time_10-30s`]: {
        min: 10000,
        max: 30000,
        mid: 20000
      },
      [`${domainPrefix}_query_to_result_time_30s+`]: {
        min: 30000,
        max: 120000,
        mid: 60000
      }
    };

    // Build cumulative distribution
    const sortedBuckets = [`${domainPrefix}_query_to_result_time_<1s`, `${domainPrefix}_query_to_result_time_1-5s`, `${domainPrefix}_query_to_result_time_5-10s`, `${domainPrefix}_query_to_result_time_10-30s`, `${domainPrefix}_query_to_result_time_30s+`];
    let totalCount = 0;
    const cumulativeCounts = [];
    for (const bucketName of sortedBuckets) {
      const count = buckets.get(bucketName) || 0;
      totalCount += count;
      cumulativeCounts.push({
        bucket: bucketName,
        count,
        cumulative: totalCount
      });
    }
    if (totalCount === 0) {
      return {
        p50: 0,
        p75: 0,
        p90: 0,
        p95: 0,
        p99: 0,
        mean: 0
      };
    }

    // Calculate percentiles using linear interpolation
    const calculatePercentile = percentile => {
      var _bucketRanges$lastBuc;
      const targetCount = percentile / 100 * totalCount;
      for (let i = 0; i < cumulativeCounts.length; i++) {
        const current = cumulativeCounts[i];
        const previous = i > 0 ? cumulativeCounts[i - 1] : {
          cumulative: 0
        };
        if (current.cumulative >= targetCount) {
          const range = bucketRanges[current.bucket];
          if (!range) return 0;

          // Linear interpolation within bucket
          const bucketProgress = current.count > 0 ? (targetCount - previous.cumulative) / current.count : 0;
          return range.min + bucketProgress * (range.max - range.min);
        }
      }

      // Fallback to last bucket
      const lastBucket = cumulativeCounts[cumulativeCounts.length - 1];
      return ((_bucketRanges$lastBuc = bucketRanges[lastBucket.bucket]) === null || _bucketRanges$lastBuc === void 0 ? void 0 : _bucketRanges$lastBuc.mid) || 0;
    };

    // Calculate mean using bucket midpoints
    let weightedSum = 0;
    for (const {
      bucket,
      count
    } of cumulativeCounts) {
      const range = bucketRanges[bucket];
      if (range) {
        weightedSum += range.mid * count;
      }
    }
    const mean = weightedSum / totalCount;
    return {
      p50: calculatePercentile(50),
      p75: calculatePercentile(75),
      p90: calculatePercentile(90),
      p95: calculatePercentile(95),
      p99: calculatePercentile(99),
      mean
    };
  }
}
exports.QueryUtils = QueryUtils;