"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAgentGraph = void 0;
var _langgraph = require("@langchain/langgraph");
var _prebuilt = require("@langchain/langgraph/prebuilt");
var _agents = require("@kbn/onechat-common/agents");
var _errors = require("@kbn/onechat-common/base/errors");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _errors2 = require("../utils/errors");
var _prompts = require("./prompts");
var _i18n = require("./i18n");
var _constants = require("./constants");
var _state = require("./state");
var _action_utils = require("./action_utils");
var _answer_agent_structured = require("./answer_agent_structured");
var _actions = require("./actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// number of successive recoverable errors we try to recover from before throwing
const MAX_ERROR_COUNT = 2;
const createAgentGraph = ({
  chatModel,
  tools,
  configuration,
  capabilities,
  logger,
  events,
  structuredOutput = false,
  outputSchema,
  processedConversation
}) => {
  const toolNode = new _prebuilt.ToolNode(tools);
  const researcherModel = chatModel.bindTools(tools).withConfig({
    tags: [_constants.tags.agent, _constants.tags.researchAgent]
  });
  const researchAgent = async state => {
    if (state.mainActions.length === 0 && state.errorCount === 0) {
      events.emit((0, _langchain.createReasoningEvent)((0, _i18n.getRandomThinkingMessage)(), {
        transient: true
      }));
    }
    try {
      const response = await researcherModel.invoke((0, _prompts.getResearchAgentPrompt)({
        customInstructions: configuration.research.instructions,
        clearSystemMessage: configuration.research.replace_default_instructions,
        capabilities,
        initialMessages: state.initialMessages,
        actions: state.mainActions,
        attachmentTypes: processedConversation.attachmentTypes
      }));
      const action = (0, _action_utils.processResearchResponse)(response);
      return {
        mainActions: [action],
        currentCycle: state.currentCycle + 1,
        errorCount: 0
      };
    } catch (error) {
      const executionError = (0, _errors2.convertError)(error);
      if ((0, _errors2.isRecoverableError)(executionError)) {
        return {
          mainActions: [(0, _actions.errorAction)(executionError)],
          errorCount: state.errorCount + 1
        };
      } else {
        throw executionError;
      }
    }
  };
  const researchAgentEdge = async state => {
    const lastAction = state.mainActions[state.mainActions.length - 1];
    if ((0, _actions.isAgentErrorAction)(lastAction)) {
      if (state.errorCount <= MAX_ERROR_COUNT) {
        return _constants.steps.researchAgent;
      } else {
        // max error count reached, stop execution by throwing
        throw lastAction.error;
      }
    } else if ((0, _actions.isToolCallAction)(lastAction)) {
      const maxCycleReached = state.currentCycle > state.cycleLimit;
      if (maxCycleReached) {
        return _constants.steps.prepareToAnswer;
      } else {
        return _constants.steps.executeTool;
      }
    } else if ((0, _actions.isHandoverAction)(lastAction)) {
      return _constants.steps.prepareToAnswer;
    }
    throw invalidState(`[researchAgentEdge] last action type was ${lastAction.type}}`);
  };
  const executeTool = async state => {
    const lastAction = state.mainActions[state.mainActions.length - 1];
    if (!(0, _actions.isToolCallAction)(lastAction)) {
      throw invalidState(`[executeTool] expected last action to be "tool_call" action, got "${lastAction.type}"`);
    }
    const toolCallMessage = (0, _langchain.createToolCallMessage)(lastAction.tool_calls, lastAction.message);
    const toolNodeResult = await toolNode.invoke([toolCallMessage], {});
    const action = (0, _action_utils.processToolNodeResponse)(toolNodeResult);
    return {
      mainActions: [action]
    };
  };
  const prepareToAnswer = async state => {
    const lastAction = state.mainActions[state.mainActions.length - 1];
    const maxCycleReached = state.currentCycle > state.cycleLimit;
    if (maxCycleReached && !(0, _actions.isHandoverAction)(lastAction)) {
      return {
        mainActions: [(0, _actions.handoverAction)('', true)]
      };
    } else {
      return {};
    }
  };
  const answeringModel = chatModel.withConfig({
    tags: [_constants.tags.agent, _constants.tags.answerAgent]
  });
  const answerAgent = async state => {
    if (state.answerActions.length === 0 && state.errorCount === 0) {
      events.emit((0, _langchain.createReasoningEvent)((0, _i18n.getRandomAnsweringMessage)(), {
        transient: true
      }));
    }
    try {
      const response = await answeringModel.invoke((0, _prompts.getAnswerAgentPrompt)({
        customInstructions: configuration.answer.instructions,
        clearSystemMessage: configuration.answer.replace_default_instructions,
        capabilities,
        initialMessages: state.initialMessages,
        actions: state.mainActions,
        answerActions: state.answerActions,
        attachmentTypes: processedConversation.attachmentTypes
      }));
      const action = (0, _action_utils.processAnswerResponse)(response);
      return {
        answerActions: [action],
        errorCount: 0
      };
    } catch (error) {
      const executionError = (0, _errors2.convertError)(error);
      if ((0, _errors2.isRecoverableError)(executionError)) {
        return {
          answerActions: [(0, _actions.errorAction)(executionError)],
          errorCount: state.errorCount + 1
        };
      } else {
        throw executionError;
      }
    }
  };
  const answerAgentStructured = (0, _answer_agent_structured.createAnswerAgentStructured)({
    chatModel,
    configuration,
    capabilities,
    events,
    outputSchema,
    attachmentTypes: processedConversation.attachmentTypes,
    logger
  });
  const answerAgentEdge = async state => {
    const lastAction = state.answerActions[state.answerActions.length - 1];
    if ((0, _actions.isAgentErrorAction)(lastAction)) {
      if (state.errorCount <= MAX_ERROR_COUNT) {
        return _constants.steps.answerAgent;
      } else {
        // max error count reached, stop execution by throwing
        throw lastAction.error;
      }
    } else if ((0, _actions.isAnswerAction)(lastAction) || (0, _actions.isStructuredAnswerAction)(lastAction)) {
      return _constants.steps.finalize;
    }

    // @ts-expect-error - lastAction.type is never because we cover all use cases.
    throw invalidState(`[answerAgentEdge] last action type was ${lastAction.type}}`);
  };
  const finalize = async state => {
    const answerAction = state.answerActions[state.answerActions.length - 1];
    if ((0, _actions.isStructuredAnswerAction)(answerAction)) {
      return {
        finalAnswer: answerAction.data
      };
    } else if ((0, _actions.isAnswerAction)(answerAction)) {
      return {
        finalAnswer: answerAction.message
      };
    } else {
      throw invalidState(`[finalize] expect answer action, got ${answerAction.type} instead.`);
    }
  };
  const selectedAnswerAgent = structuredOutput ? answerAgentStructured : answerAgent;

  // note: the node names are used in the event convertion logic, they should *not* be changed
  const graph = new _langgraph.StateGraph(_state.StateAnnotation)
  // nodes
  .addNode(_constants.steps.researchAgent, researchAgent).addNode(_constants.steps.executeTool, executeTool).addNode(_constants.steps.prepareToAnswer, prepareToAnswer).addNode(_constants.steps.answerAgent, selectedAnswerAgent).addNode(_constants.steps.finalize, finalize)
  // edges
  .addEdge(_langgraph.START, _constants.steps.researchAgent).addEdge(_constants.steps.executeTool, _constants.steps.researchAgent).addConditionalEdges(_constants.steps.researchAgent, researchAgentEdge, {
    [_constants.steps.researchAgent]: _constants.steps.researchAgent,
    [_constants.steps.executeTool]: _constants.steps.executeTool,
    [_constants.steps.prepareToAnswer]: _constants.steps.prepareToAnswer
  }).addEdge(_constants.steps.prepareToAnswer, _constants.steps.answerAgent).addConditionalEdges(_constants.steps.answerAgent, answerAgentEdge, {
    [_constants.steps.answerAgent]: _constants.steps.answerAgent,
    [_constants.steps.finalize]: _constants.steps.finalize
  }).addEdge(_constants.steps.finalize, _langgraph.END).compile();
  return graph;
};
exports.createAgentGraph = createAgentGraph;
const invalidState = message => {
  return (0, _errors.createAgentExecutionError)(message, _agents.AgentExecutionErrorCode.invalidState, {});
};