"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cleanInvalidToolReferences = cleanInvalidToolReferences;
exports.toggleToolSelection = exports.isToolSelected = void 0;
var _onechatCommon = require("@kbn/onechat-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Check if a specific tool is selected based on the current tool selections.
 * This uses existing onechat-common utilities for consistent logic.
 */
const isToolSelected = (tool, selectedTools) => {
  return selectedTools.some(selection => (0, _onechatCommon.toolMatchSelection)(tool, selection));
};

/**
 * Toggle selection for a specific tool.
 */
exports.isToolSelected = isToolSelected;
const toggleToolSelection = (toolId, allAvailableTools, selectedTools) => {
  const currentTool = {
    id: toolId
  };
  const isCurrentlySelected = isToolSelected(currentTool, selectedTools);
  if (isCurrentlySelected) {
    // Check if this tool is selected via wildcard
    const wildcardSelection = selectedTools.find(selection => selection.tool_ids.includes(_onechatCommon.allToolsSelectionWildcard));
    if (wildcardSelection) {
      // Replace wildcard with individual tool selections (excluding the one being toggled off)
      const otherToolIds = allAvailableTools.filter(tool => tool.id !== toolId).map(tool => tool.id);
      const otherSelections = selectedTools.filter(selection => !selection.tool_ids.includes(_onechatCommon.allToolsSelectionWildcard));
      if (otherToolIds.length > 0) {
        return [...otherSelections, {
          tool_ids: otherToolIds
        }];
      }
      return otherSelections;
    } else {
      // Remove from individual selections
      return selectedTools.map(selection => {
        if (selection.tool_ids.includes(toolId)) {
          const newToolIds = selection.tool_ids.filter(id => id !== toolId);
          return newToolIds.length > 0 ? {
            ...selection,
            tool_ids: newToolIds
          } : null;
        }
        return selection;
      }).filter(Boolean);
    }
  } else {
    // Add the tool to selection
    const existingSelection = selectedTools.find(selection => !selection.tool_ids.includes(_onechatCommon.allToolsSelectionWildcard));
    if (existingSelection && !existingSelection.tool_ids.includes(toolId)) {
      // Add to existing non-wildcard selection
      return selectedTools.map(selection => selection === existingSelection ? {
        ...selection,
        tool_ids: [...selection.tool_ids, toolId]
      } : selection);
    } else {
      // Create new selection
      const otherSelections = selectedTools.filter(selection => !selection.tool_ids.includes(toolId));
      return [...otherSelections, {
        tool_ids: [toolId]
      }];
    }
  }
};

/**
 * Removes invalid tool references from the agent configuration.
 * Filters out tool IDs that don't exist in the available tools list,
 * while preserving wildcard selections and removing empty selections.
 */
exports.toggleToolSelection = toggleToolSelection;
function cleanInvalidToolReferences(data, availableTools) {
  const validToolIds = new Set(availableTools.map(tool => tool.id));
  const cleanedTools = data.configuration.tools.map(selection => ({
    ...selection,
    tool_ids: selection.tool_ids.filter(id => id === '*' || validToolIds.has(id))
  })).filter(selection => selection.tool_ids.length > 0);
  return {
    ...data,
    configuration: {
      ...data.configuration,
      tools: cleanedTools
    }
  };
}