"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSubscribeToChatEvents = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _conversation = require("@kbn/onechat-common/chat/conversation");
var _rxjs = require("rxjs");
var _events = require("@kbn/onechat-common/chat/events");
var _react = require("react");
var _conversation_context = require("../conversation/conversation_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useSubscribeToChatEvents = ({
  setAgentReasoning,
  setIsResponseLoading,
  isAborted,
  browserToolExecutor
}) => {
  const {
    conversationActions,
    browserApiTools
  } = (0, _conversation_context.useConversationContext)();
  const unsubscribedRef = (0, _react.useRef)(false);
  const subscriptionRef = (0, _react.useRef)(null);
  const unsubscribeFromChatEvents = () => {
    var _subscriptionRef$curr;
    unsubscribedRef.current = true;
    (_subscriptionRef$curr = subscriptionRef.current) === null || _subscriptionRef$curr === void 0 ? void 0 : _subscriptionRef$curr.unsubscribe();
  };
  const nextChatEvent = event => {
    // chunk received, we append it to the chunk buffer
    if ((0, _onechatCommon.isMessageChunkEvent)(event)) {
      conversationActions.addAssistantMessageChunk({
        messageChunk: event.data.text_chunk
      });
    }
    // full message received, override chunk buffer
    else if ((0, _onechatCommon.isMessageCompleteEvent)(event)) {
      conversationActions.setAssistantMessage({
        assistantMessage: event.data.message_content
      });
    } else if ((0, _onechatCommon.isToolProgressEvent)(event)) {
      conversationActions.setToolCallProgress({
        progress: {
          message: event.data.message
        },
        toolCallId: event.data.tool_call_id
      });
      // Individual tool progression message should also be displayed as reasoning
      setAgentReasoning(event.data.message);
    } else if ((0, _onechatCommon.isReasoningEvent)(event)) {
      conversationActions.addReasoningStep({
        step: (0, _conversation.createReasoningStep)({
          reasoning: event.data.reasoning,
          transient: event.data.transient
        })
      });
      setAgentReasoning(event.data.reasoning);
    } else if ((0, _onechatCommon.isToolCallEvent)(event)) {
      conversationActions.addToolCall({
        step: (0, _conversation.createToolCallStep)({
          params: event.data.params,
          results: [],
          tool_call_id: event.data.tool_call_id,
          tool_id: event.data.tool_id
        })
      });
    } else if ((0, _events.isBrowserToolCallEvent)(event)) {
      // Check if this is a browser tool call and execute it immediately
      const toolId = event.data.tool_id;
      if (toolId && browserToolExecutor && browserApiTools) {
        const toolDef = browserApiTools.find(tool => tool.id === toolId);
        if (toolDef) {
          const toolsMap = new Map([[toolId, toolDef]]);
          browserToolExecutor.executeToolCalls([{
            tool_id: toolId,
            call_id: event.data.tool_call_id,
            params: event.data.params,
            timestamp: Date.now()
          }], toolsMap).catch(error => {
            // eslint-disable-next-line no-console
            console.error('Failed to execute browser tool:', error);
          });
        }
      }
    } else if ((0, _onechatCommon.isToolResultEvent)(event)) {
      const {
        tool_call_id: toolCallId,
        results
      } = event.data;
      conversationActions.setToolCallResult({
        results,
        toolCallId
      });
    } else if ((0, _onechatCommon.isRoundCompleteEvent)(event)) {
      // Now we have the full response and can stop the loading indicators
      setIsResponseLoading(false);
    } else if ((0, _onechatCommon.isConversationCreatedEvent)(event)) {
      const {
        conversation_id: id,
        title
      } = event.data;
      conversationActions.onConversationCreated({
        conversationId: id,
        title
      });
    } else if ((0, _onechatCommon.isThinkingCompleteEvent)(event)) {
      conversationActions.setTimeToFirstToken({
        timeToFirstToken: event.data.time_to_first_token
      });
    }
  };
  const subscribeToChatEvents = events$ => {
    return new Promise((resolve, reject) => {
      if (unsubscribedRef.current) {
        resolve();
        return;
      }
      subscriptionRef.current = events$.pipe((0, _rxjs.finalize)(() => {
        // When the subscription is unsubscribed from, `complete` will not be called, but the `finalize` callback will be
        if (unsubscribedRef.current) {
          resolve();
        }
      })).subscribe({
        next: nextChatEvent,
        complete: () => {
          resolve();
        },
        error: err => {
          // If the request is aborted, we don't want to show an error
          if (isAborted()) {
            resolve();
            return;
          }
          reject(err);
        }
      });
    }).finally(() => {
      if (unsubscribedRef.current) {
        unsubscribedRef.current = false;
      }
    });
  };
  return {
    subscribeToChatEvents,
    unsubscribeFromChatEvents
  };
};
exports.useSubscribeToChatEvents = useSubscribeToChatEvents;