"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexExplorer = exports.formatResource = exports.createIndexSelectorPrompt = void 0;
var _lodash = require("lodash");
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _list_search_sources = require("./steps/list_search_sources");
var _mappings = require("./utils/mappings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createIndexSummaries = async ({
  indices,
  esClient
}) => {
  const allMappings = await (0, _mappings.getIndexMappings)({
    indices: indices.map(index => index.name),
    cleanup: true,
    esClient
  });
  return indices.map(({
    name: indexName
  }) => {
    var _indexMappings$mappin;
    const indexMappings = allMappings[indexName];
    const flattened = (0, _mappings.flattenMapping)(indexMappings.mappings);
    return {
      type: _onechatCommon.EsResourceType.index,
      name: indexName,
      description: indexMappings === null || indexMappings === void 0 ? void 0 : (_indexMappings$mappin = indexMappings.mappings._meta) === null || _indexMappings$mappin === void 0 ? void 0 : _indexMappings$mappin.description,
      fields: flattened.map(field => field.path)
    };
  });
};
const createAliasSummaries = async ({
  aliases
}) => {
  // for now aliases are only described by the list of indices they target
  return aliases.map(({
    name: aliasName,
    indices
  }) => {
    return {
      type: _onechatCommon.EsResourceType.alias,
      name: aliasName,
      description: `Point to the following indices: ${indices.join(', ')}`
    };
  });
};
const createDatastreamSummaries = async ({
  datastreams,
  esClient
}) => {
  const allMappings = await (0, _mappings.getDataStreamMappings)({
    datastreams: datastreams.map(stream => stream.name),
    cleanup: true,
    esClient
  });
  return datastreams.map(({
    name
  }) => {
    var _mappings$mappings$_m;
    const mappings = allMappings[name];
    const flattened = (0, _mappings.flattenMapping)(mappings.mappings);
    return {
      type: _onechatCommon.EsResourceType.dataStream,
      name,
      description: mappings === null || mappings === void 0 ? void 0 : (_mappings$mappings$_m = mappings.mappings._meta) === null || _mappings$mappings$_m === void 0 ? void 0 : _mappings$mappings$_m.description,
      fields: flattened.map(field => field.path)
    };
  });
};
const indexExplorer = async ({
  nlQuery,
  indexPattern = '*',
  includeAliases = true,
  includeDatastream = true,
  limit = 1,
  esClient,
  model,
  logger
}) => {
  logger === null || logger === void 0 ? void 0 : logger.trace(() => `index_explorer - query="${nlQuery}", pattern="${indexPattern}"`);
  const sources = await (0, _list_search_sources.listSearchSources)({
    pattern: indexPattern,
    excludeIndicesRepresentedAsDatastream: true,
    excludeIndicesRepresentedAsAlias: false,
    esClient,
    includeKibanaIndices: indexPattern !== '*'
  });
  const indexCount = sources.indices.length;
  const aliasCount = sources.aliases.length;
  const dataStreamCount = sources.data_streams.length;
  const totalCount = indexCount + aliasCount + dataStreamCount;
  logger === null || logger === void 0 ? void 0 : logger.trace(() => `index_explorer - found ${indexCount} indices, ${aliasCount} aliases, ${dataStreamCount} datastreams for query="${nlQuery}"`);
  if (totalCount <= limit) {
    return {
      resources: [...sources.indices, ...sources.aliases, ...sources.data_streams].map(resource => {
        return {
          type: resource.type,
          name: resource.name,
          reason: `Index pattern matched less resources that the specified limit of ${limit}.`
        };
      })
    };
  }
  const resources = [];
  if (indexCount > 0) {
    const indexDescriptors = await createIndexSummaries({
      indices: sources.indices,
      esClient
    });
    resources.push(...indexDescriptors);
  }
  if (dataStreamCount > 0 && includeDatastream) {
    const dsDescriptors = await createDatastreamSummaries({
      datastreams: sources.data_streams,
      esClient
    });
    resources.push(...dsDescriptors);
  }
  if (aliasCount > 0 && includeAliases) {
    const aliasDescriptors = await createAliasSummaries({
      aliases: sources.aliases
    });
    resources.push(...aliasDescriptors);
  }
  const selectedResources = await selectResources({
    resources,
    nlQuery,
    model,
    limit
  });
  return {
    resources: selectedResources
  };
};
exports.indexExplorer = indexExplorer;
// Helper function to format each resource in an XML-like block
const formatResource = res => {
  var _res$fields, _res$description;
  const topFields = (0, _lodash.take)((_res$fields = res.fields) !== null && _res$fields !== void 0 ? _res$fields : [], 10).map(f => `      <field>${f}</field>`).join('\n');
  const description = (_res$description = res.description) !== null && _res$description !== void 0 ? _res$description : 'No description provided.';
  return `<resource type="${res.type}" name="${res.name}" description="${description}">
  <sample_fields>
${topFields || '      (No fields available)'}
  </sample_fields>
</resource>`;
};
exports.formatResource = formatResource;
const createIndexSelectorPrompt = ({
  resources,
  nlQuery,
  limit = 1
}) => {
  return [['system', `You are an AI assistant for the Elasticsearch company.

Your sole function is to identify the most relevant Elasticsearch resources (indices, aliases, data streams) based on a user's query.

You MUST call the 'select_resources' tool to provide your answer. Do NOT respond with conversational text, explanations, or any data outside of the tool call.

- The user's query will be provided.
- A list of available resources will be provided in XML format.
- You must analyze the query against the resource names, descriptions, and fields.
- Select up to a maximum of ${limit} of the most relevant resources.
- For each selected resource, you MUST provide its 'name', 'type', and a brief 'reason' for your choice.
- If NO resources are relevant, you MUST call the 'select_resources' tool with an empty 'targets' array.

Now, perform your function for the following query and resources.
`], ['human', `## Query

*The natural language query is:* "${nlQuery}"

## Available resources
<resources>
${resources.map(formatResource).join('\n')}
</resources>

Based on the natural language query and the index descriptions, please return the most relevant indices with your reasoning.
Remember, you should select at maximum ${limit} targets. If none match, just return an empty list.`]];
};
exports.createIndexSelectorPrompt = createIndexSelectorPrompt;
const selectResources = async ({
  resources,
  nlQuery,
  model,
  limit = 1
}) => {
  const {
    chatModel
  } = model;
  const indexSelectorModel = chatModel.withStructuredOutput(_zod.z.object({
    reasoning: _zod.z.string().optional().describe('optional brief overall reasoning. Can be used to explain why you did not return any target.'),
    targets: _zod.z.array(_zod.z.object({
      reason: _zod.z.string().describe('brief explanation of why this resource could be relevant'),
      type: _zod.z.enum([_onechatCommon.EsResourceType.index, _onechatCommon.EsResourceType.alias, _onechatCommon.EsResourceType.dataStream]).describe('the type of the resource'),
      name: _zod.z.string().describe('name of the index, alias or data stream')
    }))
  }).describe('Tool to use to select the relevant targets to search against'), {
    name: 'select_resources'
  });
  const promptContent = createIndexSelectorPrompt({
    resources,
    nlQuery,
    limit
  });
  const {
    targets
  } = await indexSelectorModel.invoke(promptContent);
  return targets;
};