"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.versionedAttachmentSchema = exports.versionedAttachmentInputSchema = exports.parseVersionId = exports.isAttachmentActive = exports.hashContent = exports.getVersion = exports.getLatestVersion = exports.getActiveAttachments = exports.estimateTokens = exports.createVersionId = exports.attachmentVersionSchema = exports.attachmentVersionRefSchema = exports.attachmentDiffSchema = void 0;
var _zod = require("@kbn/zod");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Represents a single version of an attachment's content.
 */

/**
 * Represents a conversation-level versioned attachment.
 * Contains all versions of the attachment and metadata.
 */

/**
 * Reference to a specific version of an attachment.
 * Used in RoundInput to reference conversation-level attachments.
 */

/**
 * Represents a diff between two versions of an attachment.
 */

/**
 * Input for creating a new versioned attachment.
 */

// Zod schemas for validation

const attachmentVersionRefSchema = exports.attachmentVersionRefSchema = _zod.z.object({
  attachment_id: _zod.z.string(),
  version: _zod.z.number().int().positive()
});
const attachmentVersionSchema = exports.attachmentVersionSchema = _zod.z.object({
  version: _zod.z.number().int().positive(),
  data: _zod.z.unknown(),
  created_at: _zod.z.string(),
  content_hash: _zod.z.string(),
  estimated_tokens: _zod.z.number().int().optional()
});
const versionedAttachmentSchema = exports.versionedAttachmentSchema = _zod.z.object({
  id: _zod.z.string(),
  type: _zod.z.string(),
  versions: _zod.z.array(attachmentVersionSchema),
  current_version: _zod.z.number().int().positive(),
  description: _zod.z.string().optional(),
  active: _zod.z.boolean().optional(),
  hidden: _zod.z.boolean().optional(),
  client_id: _zod.z.string().optional()
});
const versionedAttachmentInputSchema = exports.versionedAttachmentInputSchema = _zod.z.object({
  id: _zod.z.string().optional(),
  type: _zod.z.string(),
  data: _zod.z.unknown(),
  description: _zod.z.string().optional(),
  hidden: _zod.z.boolean().optional()
});
const attachmentDiffSchema = exports.attachmentDiffSchema = _zod.z.object({
  change_type: _zod.z.enum(['create', 'update', 'delete', 'restore']),
  summary: _zod.z.string(),
  changed_fields: _zod.z.array(_zod.z.string()).optional()
});

// Utility functions

/**
 * Gets the latest (current) version of an attachment.
 */
const getLatestVersion = attachment => {
  return attachment.versions.find(v => v.version === attachment.current_version);
};

/**
 * Gets a specific version of an attachment.
 */
exports.getLatestVersion = getLatestVersion;
const getVersion = (attachment, version) => {
  return attachment.versions.find(v => v.version === version);
};

/**
 * Creates a unique identifier for a specific attachment version.
 */
exports.getVersion = getVersion;
const createVersionId = (attachmentId, version) => {
  return `${attachmentId}:v${version}`;
};

/**
 * Parses a version ID back into its components.
 */
exports.createVersionId = createVersionId;
const parseVersionId = versionId => {
  const match = versionId.match(/^(.+):v(\d+)$/);
  if (!match) {
    return undefined;
  }
  return {
    attachmentId: match[1],
    version: parseInt(match[2], 10)
  };
};

/**
 * Checks if an attachment's current version is active (not deleted).
 */
exports.parseVersionId = parseVersionId;
const isAttachmentActive = attachment => {
  return attachment.active !== false;
};

/**
 * Gets all active (non-deleted) attachments from a list.
 */
exports.isAttachmentActive = isAttachmentActive;
const getActiveAttachments = attachments => {
  return attachments.filter(isAttachmentActive);
};

/**
 * Simple hash function for content deduplication.
 * Uses a basic string hash - suitable for detecting duplicates.
 */
exports.getActiveAttachments = getActiveAttachments;
const hashContent = data => {
  const str = JSON.stringify(data);
  let hash = 0;
  for (let i = 0; i < str.length; i++) {
    const char = str.charCodeAt(i);
    // eslint-disable-next-line no-bitwise
    hash = (hash << 5) - hash + char;
    // eslint-disable-next-line no-bitwise
    hash = hash & hash; // Convert to 32bit integer
  }
  return hash.toString(16);
};

/**
 * Estimates token count for attachment data.
 * Uses a simple heuristic: ~4 characters per token.
 */
exports.hashContent = hashContent;
const estimateTokens = data => {
  const str = JSON.stringify(data);
  return Math.ceil(str.length / 4);
};
exports.estimateTokens = estimateTokens;