"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AlertData = void 0;
var _lodash = require("lodash");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// TS will make sure that if we add a new supported rule type id we had the corresponding function to get the relevant rule fields
const getRelevantRuleFieldsMap = {
  [_ruleDataUtils.OBSERVABILITY_THRESHOLD_RULE_TYPE_ID]: customThresholdParams => {
    const relevantFields = new Set();
    const metrics = customThresholdParams.criteria[0].metrics;
    metrics.forEach(metric => {
      // The property "field" is of type string | never but it collapses to just string
      // We should probably avoid typing field as never and just omit it from the type to avoid situations like this one
      if ('field' in metric) relevantFields.add(metric.field);
    });
    return relevantFields;
  }
};
const getRuleQueryIndexMap = {
  [_ruleDataUtils.OBSERVABILITY_THRESHOLD_RULE_TYPE_ID]: customThresholdParams => {
    const {
      searchConfiguration: {
        index
      }
    } = customThresholdParams;
    if (typeof index === 'object') return (index === null || index === void 0 ? void 0 : index.id) || null;
    if (typeof index === 'string') return index;
    return null;
  }
};
class AlertData {
  constructor(alert) {
    this.alert = alert;
  }
  getRuleParameters() {
    return this.alert[_ruleDataUtils.ALERT_RULE_PARAMETERS];
  }
  getRuleId() {
    return this.alert[_ruleDataUtils.ALERT_RULE_UUID];
  }
  getRelevantRuleFields() {
    const ruleParameters = this.getRuleParameters();
    if (!ruleParameters) {
      throw new Error('No rule parameters found');
    }
    const ruleTypeId = this.getRuleTypeId();
    return (0, _helpers.isSuggestedDashboardsValidRuleTypeId)(ruleTypeId) ? getRelevantRuleFieldsMap[ruleTypeId](ruleParameters) : new Set();
  }
  getRelevantAADFields() {
    const ignoredFields = ['_index'];
    const allKibanaFields = Object.keys(this.alert).filter(field => field.startsWith('kibana.'));
    const nonTechnicalFields = (0, _lodash.omit)(this.alert, [...Object.values(_ruleDataUtils.fields), ...allKibanaFields, ...ignoredFields]);
    return Object.keys(nonTechnicalFields);
  }
  getAllRelevantFields() {
    const ruleFields = this.getRelevantRuleFields();
    const aadFields = this.getRelevantAADFields();
    return Array.from(new Set([...ruleFields, ...aadFields]));
  }
  getAlertTags() {
    return this.alert.tags || [];
  }
  getRuleQueryIndex() {
    const ruleParameters = this.getRuleParameters();
    const ruleTypeId = this.getRuleTypeId();
    if (!ruleParameters) {
      throw new Error('No rule parameters found');
    }
    return (0, _helpers.isSuggestedDashboardsValidRuleTypeId)(ruleTypeId) ? getRuleQueryIndexMap[ruleTypeId](ruleParameters) : null;
  }
  getRuleTypeId() {
    return this.alert[_ruleDataUtils.ALERT_RULE_TYPE_ID];
  }
}
exports.AlertData = AlertData;