"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mapRuleParamsWithFlyout = void 0;
var _alertingComparators = require("@kbn/alerting-comparators");
var _i18n = require("@kbn/i18n");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _common = require("../../../../common");
var _formatters = require("../../../../common/custom_threshold_rule/formatters");
var _snapshot_metric_formats = require("../../../../common/custom_threshold_rule/formatters/snapshot_metric_formats");
var _metric_value_formatter = require("../../../../common/custom_threshold_rule/metric_value_formatter");
var _i18n2 = require("../../../../common/i18n");
var _is_fields_same_type = require("./is_fields_same_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getI18nComparator = comparator => {
  switch (comparator) {
    case _alertingComparators.COMPARATORS.GREATER_THAN:
      return _i18n2.ABOVE_TEXT;
    case _alertingComparators.COMPARATORS.GREATER_THAN_OR_EQUALS:
      return _i18n2.ABOVE_OR_EQ_TEXT;
    case _alertingComparators.COMPARATORS.LESS_THAN:
      return _i18n2.BELOW_TEXT;
    case _alertingComparators.COMPARATORS.LESS_THAN_OR_EQUALS:
      return _i18n2.BELOW_OR_EQ_TEXT;
    default:
      return comparator;
  }
};
const getPctAboveThreshold = (threshold, comparator, observedValue, isThereAWarningThreshold) => {
  if (!observedValue || !threshold || threshold.length > 1 || threshold[0] <= 0 || isThereAWarningThreshold === true) return '';
  return _i18n.i18n.translate('xpack.observability.alertFlyout.overview.aboveThresholdLabel', {
    defaultMessage: ' ({pctValue}% {comparator} the threshold)',
    values: {
      pctValue: Math.abs(parseFloat(((observedValue - threshold[0]) * 100 / threshold[0]).toFixed(2))),
      comparator: getI18nComparator((0, _common.convertToBuiltInComparators)(comparator))
    }
  });
};

// Helper function to convert threshold values (divide by 100 for percent, by 8 for bits)
const convertThresholdValues = (threshold, metricField, metric) => {
  return threshold.map(thresholdToFormat => {
    if (metricField.endsWith('.pct') || _snapshot_metric_formats.METRIC_FORMATTERS[metric] && _snapshot_metric_formats.METRIC_FORMATTERS[metric].formatter === 'percent') {
      thresholdToFormat = thresholdToFormat / 100;
    } else if (metricField.endsWith('.bytes') || _snapshot_metric_formats.METRIC_FORMATTERS[metric] && _snapshot_metric_formats.METRIC_FORMATTERS[metric].formatter === 'bits') {
      thresholdToFormat = thresholdToFormat / 8;
    }
    return thresholdToFormat;
  });
};

// Helper function to format threshold values using metricValueFormatter
const formatThresholdValues = (threshold, field) => {
  return threshold.map(thresholdValue => (0, _metric_value_formatter.metricValueFormatter)(thresholdValue, field)).join(' AND ');
};

// Helper function to format inventory value based on custom metric field
const formatInventoryValue = (value, customMetricField, metric) => {
  if (customMetricField) {
    return (0, _metric_value_formatter.metricValueFormatter)(value, customMetricField);
  } else {
    const infraType = _snapshot_metric_formats.METRIC_FORMATTERS[metric].formatter;
    const formatter = (0, _formatters.createFormatter)(infraType);
    return formatter(value);
  }
};

// Helper function to format inventory threshold based on custom metric field
const formatInventoryThreshold = (threshold, customMetricField, metricField, metric) => {
  if (customMetricField) {
    return threshold.map(thresholdValue => (0, _metric_value_formatter.metricValueFormatter)(thresholdValue, metricField)).join(' AND ');
  } else {
    const thresholdFormatted = convertThresholdValues(threshold, metricField, metric);
    const infraType = _snapshot_metric_formats.METRIC_FORMATTERS[metric].formatter;
    const formatter = (0, _formatters.createFormatter)(infraType);
    return thresholdFormatted.map(formatter).join(' AND ');
  }
};
const mapRuleParamsWithFlyout = alert => {
  const ruleParams = alert.fields[_ruleDataUtils.ALERT_RULE_PARAMETERS];
  if (!ruleParams) return;
  const ruleCriteria = ruleParams === null || ruleParams === void 0 ? void 0 : ruleParams.criteria;
  const ruleId = alert.fields[_ruleDataUtils.ALERT_RULE_TYPE_ID];
  const observedValues = alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUES] || [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]];
  switch (ruleId) {
    case _ruleDataUtils.OBSERVABILITY_THRESHOLD_RULE_TYPE_ID:
      return observedValues.map((observedValue, metricIndex) => {
        const criteria = Array.isArray(ruleCriteria) ? ruleCriteria[metricIndex] : ruleCriteria;
        const fields = criteria.metrics.map(metric => metric.field || 'COUNT_AGG');
        const comparator = criteria.comparator;
        const threshold = criteria.threshold;
        const isSameFieldsType = (0, _is_fields_same_type.isFieldsSameType)(fields);
        const formattedValue = (0, _metric_value_formatter.metricValueFormatter)(observedValue, isSameFieldsType ? fields[0] : 'noType');
        const thresholdFormattedAsString = formatThresholdValues(threshold, isSameFieldsType ? fields[0] : 'noType');
        return {
          observedValue: formattedValue,
          threshold: thresholdFormattedAsString,
          comparator,
          pctAboveThreshold: getPctAboveThreshold(threshold, (0, _common.convertToBuiltInComparators)(comparator), observedValue)
        };
      });
    case _ruleDataUtils.METRIC_THRESHOLD_ALERT_TYPE_ID:
      return observedValues.map((observedValue, metricIndex) => {
        const criteria = Array.isArray(ruleCriteria) ? ruleCriteria[metricIndex] : ruleCriteria;
        let fields = [];
        const metric = criteria.metric;
        const customMetric = criteria.customMetrics;
        if (metric) fields = [metric];
        if (customMetric && customMetric.length) fields = customMetric.map(cMetric => cMetric.field);
        const comparator = criteria.comparator;
        const threshold = criteria.threshold;
        const isSameFieldsType = (0, _is_fields_same_type.isFieldsSameType)(fields);
        const formattedValue = (0, _metric_value_formatter.metricValueFormatter)(observedValue, isSameFieldsType ? fields[0] : 'noType');
        const thresholdFormattedAsString = formatThresholdValues(threshold, isSameFieldsType ? fields[0] : 'noType');
        const result = {
          observedValue: formattedValue,
          threshold: thresholdFormattedAsString,
          comparator,
          pctAboveThreshold: getPctAboveThreshold(threshold, (0, _common.convertToBuiltInComparators)(comparator), observedValue, criteria.warningThreshold !== undefined && criteria.warningComparator !== undefined ? true : false)
        };
        if (criteria.warningThreshold && criteria.warningComparator) {
          const warningThresholdFormattedAsString = formatThresholdValues(criteria.warningThreshold, isSameFieldsType ? fields[0] : 'noType');
          result.warningThreshold = warningThresholdFormattedAsString;
          result.warningComparator = criteria.warningComparator;
        }
        return result;
      });
    case _ruleDataUtils.METRIC_INVENTORY_THRESHOLD_ALERT_TYPE_ID:
      return observedValues.map((observedValue, metricIndex) => {
        const criteria = Array.isArray(ruleCriteria) ? ruleCriteria[metricIndex] : ruleCriteria;
        const {
          threshold,
          customMetric,
          metric,
          comparator
        } = criteria;
        const metricField = (customMetric === null || customMetric === void 0 ? void 0 : customMetric.field) || metric;
        const thresholdFormatted = convertThresholdValues(threshold, metricField, metric);
        const observedValueFormatted = formatInventoryValue(observedValue, customMetric.field, metric);
        const thresholdFormattedAsString = formatInventoryThreshold(threshold, customMetric.field, metricField, metric);
        const result = {
          observedValue: observedValueFormatted,
          threshold: thresholdFormattedAsString,
          comparator,
          pctAboveThreshold: getPctAboveThreshold(thresholdFormatted, (0, _common.convertToBuiltInComparators)(comparator), observedValue, criteria.warningThreshold !== undefined && criteria.warningComparator !== undefined ? true : false)
        };

        // If warning thresholds exist, add them to the same entry
        // Use the same formatting approach as critical threshold
        if (criteria.warningThreshold && criteria.warningComparator) {
          result.warningThreshold = formatInventoryThreshold(criteria.warningThreshold, customMetric.field, metricField, metric);
          result.warningComparator = criteria.warningComparator;
        }
        return result;
      });
    case _ruleDataUtils.LOG_THRESHOLD_ALERT_TYPE_ID:
      const {
        comparator
      } = ruleParams === null || ruleParams === void 0 ? void 0 : ruleParams.count;
      const flyoutMap = {
        observedValue: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]],
        threshold: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]],
        comparator,
        pctAboveThreshold: getPctAboveThreshold([alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]], comparator, alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE])
      };
      return [flyoutMap];
    case _ruleDataUtils.ApmRuleType.ErrorCount:
      const APMFlyoutMapErrorCount = {
        observedValue: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]],
        threshold: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]],
        comparator: _alertingComparators.COMPARATORS.GREATER_THAN,
        pctAboveThreshold: getPctAboveThreshold([alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]], _alertingComparators.COMPARATORS.GREATER_THAN, alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE])
      };
      return [APMFlyoutMapErrorCount];
    case _ruleDataUtils.ApmRuleType.TransactionErrorRate:
      const APMFlyoutMapTransactionErrorRate = {
        observedValue: [(0, _common.asPercent)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], 100)],
        threshold: [(0, _common.asPercent)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD], 100)],
        comparator: _alertingComparators.COMPARATORS.GREATER_THAN,
        pctAboveThreshold: getPctAboveThreshold([alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]], _alertingComparators.COMPARATORS.GREATER_THAN, alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE])
      };
      return [APMFlyoutMapTransactionErrorRate];
    case _ruleDataUtils.ApmRuleType.TransactionDuration:
      const APMFlyoutMapTransactionDuration = {
        observedValue: [(0, _common.asDuration)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE])],
        threshold: [(0, _common.asDuration)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD])],
        comparator: _alertingComparators.COMPARATORS.GREATER_THAN,
        pctAboveThreshold: getPctAboveThreshold([alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]], _alertingComparators.COMPARATORS.GREATER_THAN, alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE])
      };
      return [APMFlyoutMapTransactionDuration];
    case '.es-query':
      const {
        thresholdComparator,
        threshold
      } = ruleParams;
      const ESQueryFlyoutMap = {
        observedValue: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]],
        threshold: [threshold].flat().join(' AND '),
        comparator: thresholdComparator,
        pctAboveThreshold: getPctAboveThreshold(threshold, thresholdComparator, alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE])
      };
      return [ESQueryFlyoutMap];
    case _ruleDataUtils.SLO_BURN_RATE_RULE_TYPE_ID:
      const SLOBurnRateFlyoutMap = {
        observedValue: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE]],
        threshold: [alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]],
        comparator: _alertingComparators.COMPARATORS.GREATER_THAN,
        pctAboveThreshold: getPctAboveThreshold([alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]], _alertingComparators.COMPARATORS.GREATER_THAN, alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE])
      };
      return [SLOBurnRateFlyoutMap];
    case _ruleDataUtils.DEGRADED_DOCS_RULE_TYPE_ID:
      const DegradedDocsFlyoutMap = {
        observedValue: [(0, _common.asPercent)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_VALUE], 100)],
        threshold: [(0, _common.asPercent)(alert.fields[_ruleDataUtils.ALERT_EVALUATION_THRESHOLD], 100)],
        comparator: ruleParams.comparator
      };
      return [DegradedDocsFlyoutMap];
    default:
      return [];
  }
};
exports.mapRuleParamsWithFlyout = mapRuleParamsWithFlyout;