"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.functionRoutes = void 0;
var _ioTsUtils = require("@kbn/io-ts-utils");
var t = _interopRequireWildcard(require("io-ts"));
var _uuid = require("uuid");
var _types = require("../../../common/types");
var _get_system_message_from_instructions = require("../../service/util/get_system_message_from_instructions");
var _create_observability_ai_assistant_server_route = require("../create_observability_ai_assistant_server_route");
var _runtime_types = require("../runtime_types");
var _get_dataset_info = require("../../functions/get_dataset_info");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFunctionsRoute = (0, _create_observability_ai_assistant_server_route.createObservabilityAIAssistantServerRoute)({
  endpoint: 'GET /internal/observability_ai_assistant/functions',
  params: t.type({
    query: t.partial({
      scopes: t.union([t.array(_runtime_types.assistantScopeType), _runtime_types.assistantScopeType])
    })
  }),
  security: {
    authz: {
      requiredPrivileges: ['ai_assistant']
    }
  },
  handler: async resources => {
    const {
      service,
      request,
      params: {
        query: {
          scopes: inputScopes
        }
      }
    } = resources;
    const scopes = inputScopes ? Array.isArray(inputScopes) ? inputScopes : [inputScopes] : [];
    const controller = new AbortController();
    request.events.aborted$.subscribe(() => {
      controller.abort();
    });
    const client = await service.getClient({
      request
    });
    const [functionClient, kbUserInstructions] = await Promise.all([service.getFunctionClient({
      signal: controller.signal,
      resources,
      client,
      screenContexts: [],
      scopes
    }),
    // error is caught in client
    client.getKnowledgeBaseUserInstructions()]);
    const functionDefinitions = functionClient.getFunctions().map(fn => fn.definition);
    const availableFunctionNames = functionDefinitions.map(def => def.name);
    return {
      functionDefinitions,
      systemMessage: (0, _get_system_message_from_instructions.getSystemMessageFromInstructions)({
        applicationInstructions: functionClient.getInstructions(),
        kbUserInstructions,
        apiUserInstructions: [],
        availableFunctionNames
      })
    };
  }
});
const functionDatasetInfoRoute = (0, _create_observability_ai_assistant_server_route.createObservabilityAIAssistantServerRoute)({
  endpoint: 'GET /internal/observability_ai_assistant/functions/get_dataset_info',
  params: t.type({
    query: t.type({
      index: t.string,
      connectorId: t.string
    })
  }),
  security: {
    authz: {
      requiredPrivileges: ['ai_assistant']
    }
  },
  handler: async resources => {
    const client = await resources.service.getClient({
      request: resources.request
    });
    const {
      query: {
        index,
        connectorId
      }
    } = resources.params;
    const controller = new AbortController();
    resources.request.events.aborted$.subscribe(() => {
      controller.abort();
    });
    const resp = await (0, _get_dataset_info.getDatasetInfo)({
      resources,
      indexPattern: index,
      signal: controller.signal,
      messages: [],
      chat: (operationName, params) => {
        return client.chat(operationName, {
          ...params,
          stream: true,
          connectorId
        });
      }
    });
    return resp;
  }
});
const functionRecallRoute = (0, _create_observability_ai_assistant_server_route.createObservabilityAIAssistantServerRoute)({
  endpoint: 'POST /internal/observability_ai_assistant/functions/recall',
  params: t.type({
    body: t.intersection([t.type({
      queries: t.array(t.intersection([t.type({
        text: t.string
      }), t.partial({
        boost: t.number
      })]))
    }), t.partial({
      categories: t.array(t.string)
    })])
  }),
  security: {
    authz: {
      requiredPrivileges: ['ai_assistant']
    }
  },
  handler: async resources => {
    const client = await resources.service.getClient({
      request: resources.request
    });
    const {
      body: {
        queries,
        categories
      }
    } = resources.params;
    const entries = await client.recall({
      queries,
      categories
    });
    return {
      entries
    };
  }
});
const functionSummariseRoute = (0, _create_observability_ai_assistant_server_route.createObservabilityAIAssistantServerRoute)({
  endpoint: 'POST /internal/observability_ai_assistant/functions/summarize',
  params: t.type({
    body: t.type({
      title: t.string,
      text: _ioTsUtils.nonEmptyStringRt,
      public: _ioTsUtils.toBooleanRt,
      labels: t.record(t.string, t.string)
    })
  }),
  security: {
    authz: {
      requiredPrivileges: ['ai_assistant']
    }
  },
  handler: async resources => {
    const client = await resources.service.getClient({
      request: resources.request
    });
    const {
      title,
      text,
      public: isPublic,
      labels
    } = resources.params.body;
    return client.addKnowledgeBaseEntry({
      entry: {
        title,
        id: (0, _uuid.v4)(),
        text,
        public: isPublic,
        labels,
        role: _types.KnowledgeBaseEntryRole.AssistantSummarization
      }
    });
  }
});
const functionRoutes = exports.functionRoutes = {
  ...getFunctionsRoute,
  ...functionRecallRoute,
  ...functionSummariseRoute,
  ...functionDatasetInfoRoute
};