"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.chatRoutes = void 0;
var _boom = require("@hapi/boom");
var _ioTsUtils = require("@kbn/io-ts-utils");
var t = _interopRequireWildcard(require("io-ts"));
var _rxjs = require("rxjs");
var _uuid = require("uuid");
var _lodash = require("lodash");
var _ = require("../..");
var _create_function_response_message = require("../../../common/utils/create_function_response_message");
var _flush_buffer = require("../../service/util/flush_buffer");
var _observable_into_openai_stream = require("../../service/util/observable_into_openai_stream");
var _observable_into_stream = require("../../service/util/observable_into_stream");
var _with_assistant_span = require("../../service/util/with_assistant_span");
var _recall_and_score = require("../../functions/context/utils/recall_and_score");
var _create_observability_ai_assistant_server_route = require("../create_observability_ai_assistant_server_route");
var _runtime_types = require("../runtime_types");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const chatCompleteBaseRt = apiType => t.type({
  body: t.intersection([t.type({
    messages: t.array(apiType === 'public' ? _runtime_types.publicMessageRt : _runtime_types.messageRt),
    connectorId: t.string,
    persist: _ioTsUtils.toBooleanRt
  }), t.partial({
    isStream: t.union([t.undefined, _ioTsUtils.toBooleanRt]),
    // accept undefined in order to default to true
    conversationId: t.string,
    title: t.string,
    disableFunctions: _ioTsUtils.toBooleanRt,
    instructions: t.array(t.union([t.string, t.type({
      id: t.string,
      text: t.string
    })]))
  })])
});
const chatCompleteInternalRt = t.intersection([chatCompleteBaseRt('internal'), t.type({
  body: t.type({
    screenContexts: t.array(_runtime_types.screenContextRt),
    scopes: t.array(_runtime_types.assistantScopeType)
  })
})]);
const chatCompletePublicRt = t.intersection([chatCompleteBaseRt('public'), t.partial({
  body: t.partial({
    actions: t.array(_runtime_types.functionRt)
  })
})]);
async function initializeChatRequest({
  context,
  request,
  plugins: {
    cloud,
    actions
  },
  params: {
    body: {
      connectorId,
      scopes
    }
  },
  service
}) {
  await (0, _with_assistant_span.withAssistantSpan)('guard_against_invalid_connector', async () => {
    const actionsClient = await (await actions.start()).getActionsClientWithRequest(request);
    const connector = await actionsClient.get({
      id: connectorId,
      throwIfSystemAction: true
    });
    return connector;
  });
  const [client, cloudStart, simulateFunctionCalling] = await Promise.all([service.getClient({
    request,
    scopes
  }), cloud === null || cloud === void 0 ? void 0 : cloud.start(), (await context.core).uiSettings.client.get(_.aiAssistantSimulatedFunctionCalling)]);
  if (!client) {
    throw (0, _boom.notImplemented)();
  }
  const controller = new AbortController();
  request.events.aborted$.subscribe(() => {
    controller.abort();
  });
  return {
    client,
    isCloudEnabled: Boolean(cloudStart === null || cloudStart === void 0 ? void 0 : cloudStart.isCloudEnabled),
    simulateFunctionCalling,
    signal: controller.signal
  };
}
const chatRoute = (0, _create_observability_ai_assistant_server_route.createObservabilityAIAssistantServerRoute)({
  endpoint: 'POST /internal/observability_ai_assistant/chat',
  security: {
    authz: {
      requiredPrivileges: ['ai_assistant']
    }
  },
  params: t.type({
    body: t.intersection([t.type({
      name: t.string,
      systemMessage: t.string,
      messages: t.array(_runtime_types.messageRt),
      connectorId: t.string,
      functions: t.array(_runtime_types.functionRt),
      scopes: t.array(_runtime_types.assistantScopeType)
    }), t.partial({
      functionCall: t.string
    })])
  }),
  handler: async resources => {
    const {
      params
    } = resources;
    const {
      body: {
        name,
        systemMessage,
        messages,
        connectorId,
        functions,
        functionCall
      }
    } = params;
    const {
      client,
      simulateFunctionCalling,
      signal,
      isCloudEnabled
    } = await initializeChatRequest(resources);
    const response$ = client.chat(name, {
      stream: true,
      systemMessage,
      messages,
      connectorId,
      signal,
      ...(functions.length ? {
        functions,
        functionCall
      } : {}),
      simulateFunctionCalling
    });
    return (0, _observable_into_stream.observableIntoStream)(response$.pipe((0, _flush_buffer.flushBuffer)(isCloudEnabled)));
  }
});
const chatRecallRoute = (0, _create_observability_ai_assistant_server_route.createObservabilityAIAssistantServerRoute)({
  endpoint: 'POST /internal/observability_ai_assistant/chat/recall',
  security: {
    authz: {
      requiredPrivileges: ['ai_assistant']
    }
  },
  params: t.type({
    body: t.type({
      screenDescription: t.string,
      connectorId: t.string,
      scopes: t.array(_runtime_types.assistantScopeType),
      messages: t.array(_runtime_types.messageRt)
    })
  }),
  handler: async resources => {
    const {
      client,
      simulateFunctionCalling,
      signal,
      isCloudEnabled
    } = await initializeChatRequest(resources);
    const {
      request,
      plugins
    } = resources;
    const actionsClient = await (await plugins.actions.start()).getActionsClientWithRequest(request);
    const {
      connectorId,
      screenDescription,
      messages,
      scopes
    } = resources.params.body;
    const connector = await actionsClient.get({
      id: connectorId,
      throwIfSystemAction: true
    });
    const response$ = (0, _rxjs.from)((0, _recall_and_score.recallAndScore)({
      analytics: (await resources.plugins.core.start()).analytics,
      chat: (name, params) => client.chat(name, {
        ...params,
        stream: true,
        connectorId,
        simulateFunctionCalling,
        signal
      }),
      screenDescription,
      logger: resources.logger,
      messages,
      recall: client.recall,
      signal,
      connector,
      scopes
    })).pipe((0, _rxjs.map)(({
      llmScores,
      suggestions,
      relevantDocuments
    }) => {
      return (0, _create_function_response_message.createFunctionResponseMessage)({
        name: 'context',
        data: {
          suggestions,
          llmScores
        },
        content: {
          relevantDocuments
        }
      });
    }));
    return (0, _observable_into_stream.observableIntoStream)(response$.pipe((0, _flush_buffer.flushBuffer)(isCloudEnabled)));
  }
});
async function chatComplete(resources) {
  const {
    params,
    service
  } = resources;
  const {
    body: {
      messages,
      connectorId,
      conversationId,
      title,
      persist,
      screenContexts,
      instructions: userInstructionsOrStrings,
      disableFunctions,
      scopes
    }
  } = params;
  resources.logger.debug(`Initializing chat request with ${messages.length} messages`);
  const {
    client,
    isCloudEnabled,
    signal,
    simulateFunctionCalling
  } = await initializeChatRequest(resources);
  const functionClient = await service.getFunctionClient({
    signal,
    resources,
    client,
    screenContexts,
    scopes
  });
  const userInstructions = userInstructionsOrStrings === null || userInstructionsOrStrings === void 0 ? void 0 : userInstructionsOrStrings.map(userInstructionOrString => typeof userInstructionOrString === 'string' ? {
    text: userInstructionOrString,
    id: (0, _uuid.v4)()
  } : userInstructionOrString);
  const {
    response$,
    getConversation
  } = client.complete({
    messages,
    connectorId,
    conversationId,
    title,
    persist,
    signal,
    functionClient,
    userInstructions,
    simulateFunctionCalling,
    disableFunctions
  });
  const responseWithFlushBuffer$ = response$.pipe((0, _flush_buffer.flushBuffer)(isCloudEnabled));
  return {
    response$: responseWithFlushBuffer$,
    getConversation
  };
}
const chatCompleteRoute = (0, _create_observability_ai_assistant_server_route.createObservabilityAIAssistantServerRoute)({
  endpoint: 'POST /internal/observability_ai_assistant/chat/complete',
  security: {
    authz: {
      requiredPrivileges: ['ai_assistant']
    }
  },
  params: chatCompleteInternalRt,
  handler: async resources => {
    const {
      params
    } = resources;
    const {
      response$,
      getConversation
    } = await chatComplete(resources);
    const {
      isStream = true,
      connectorId
    } = params.body;
    if (isStream === false) {
      var _last;
      const response = await getConversation();
      return {
        conversationId: response.conversation.id,
        data: (_last = (0, _lodash.last)(response.messages)) === null || _last === void 0 ? void 0 : _last.message.content,
        connectorId
      };
    }
    return (0, _observable_into_stream.observableIntoStream)(response$);
  }
});
const publicChatCompleteRoute = (0, _create_observability_ai_assistant_server_route.createObservabilityAIAssistantServerRoute)({
  endpoint: 'POST /api/observability_ai_assistant/chat/complete 2023-10-31',
  security: {
    authz: {
      requiredPrivileges: ['ai_assistant']
    }
  },
  params: chatCompletePublicRt,
  options: {
    tags: ['observability-ai-assistant']
  },
  handler: async resources => {
    const {
      params,
      logger
    } = resources;
    const {
      actions,
      ...bodyParams
    } = params.body;
    const {
      response$
    } = await chatComplete({
      ...resources,
      params: {
        body: {
          ...bodyParams,
          scopes: ['observability'],
          screenContexts: [{
            actions
          }]
        }
      }
    });
    return (0, _observable_into_openai_stream.observableIntoOpenAIStream)(response$, logger);
  }
});
const chatRoutes = exports.chatRoutes = {
  ...chatRoute,
  ...chatRecallRoute,
  ...chatCompleteRoute,
  ...publicChatCompleteRoute
};