"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_SERVICES_TOOL_ID = void 0;
exports.createGetServicesTool = createGetServicesTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _tool_schemas = require("../../utils/tool_schemas");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_SERVICES_TOOL_ID = exports.OBSERVABILITY_GET_SERVICES_TOOL_ID = 'observability.get_services';
const getServicesSchema = _zod.z.object({
  ..._tool_schemas.timeRangeSchemaRequired,
  environment: _zod.z.string().min(1).optional().describe('Optionally filter the services by the environments that they are running in.'),
  healthStatus: _zod.z.array(_zod.z.enum(['unknown', 'healthy', 'warning', 'critical'])).optional().describe('Optionally filter the services by their health status.')
});
function createGetServicesTool({
  core,
  dataRegistry,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_SERVICES_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: 'Retrieves a list of monitored APM services, including their health status, active alert counts, and key performance metrics: latency, transaction error rate, and throughput. Useful for high-level system overview, identifying unhealthy services, and quantifying performance issues.',
    schema: getServicesSchema,
    tags: ['observability', 'services'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async ({
      start,
      end,
      environment,
      healthStatus
    }, context) => {
      const {
        request
      } = context;
      try {
        const {
          services,
          maxCountExceeded,
          serviceOverflowCount
        } = await (0, _handler.getToolHandler)({
          request,
          dataRegistry,
          start,
          end,
          environment,
          healthStatus
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              services,
              maxCountExceeded,
              serviceOverflowCount
            }
          }]
        };
      } catch (error) {
        logger.error(`Error getting services: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch services: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}