"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_LOG_CATEGORIES_TOOL_ID = void 0;
exports.createGetLogCategoriesTool = createGetLogCategoriesTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _tool_schemas = require("../../utils/tool_schemas");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_TIME_RANGE = {
  start: 'now-1h',
  end: 'now'
};
const OBSERVABILITY_GET_LOG_CATEGORIES_TOOL_ID = exports.OBSERVABILITY_GET_LOG_CATEGORIES_TOOL_ID = 'observability.get_log_categories';
const getLogsSchema = _zod.z.object({
  ...(0, _tool_schemas.timeRangeSchemaOptional)(DEFAULT_TIME_RANGE),
  index: _zod.z.string().describe(_tool_schemas.indexDescription).optional(),
  terms: _zod.z.record(_zod.z.string(), _zod.z.string()).optional().describe('Optional field filters to narrow down results. Each key-value pair filters logs where the field exactly matches the value. Example: { "service.name": "payment", "host.name": "web-server-01" }. Multiple filters are combined with AND logic.')
});
function createGetLogCategoriesTool({
  core,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_LOG_CATEGORIES_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Compresses thousands of logs into a small set of categories to provide a high-level overview of what's being logged.

When to use:
- Getting a quick summary of log activity in a service or time range
- Identifying the types of events occurring without reading individual logs
- Discovering unexpected log patterns or new error types
- Answering "what kinds of things are happening?" rather than "what exactly happened?"

How it works:
Groups similar log messages together using pattern recognition, returning representative categories with counts.

Do NOT use for:
- Understanding the sequence of events for a specific error (use get_correlated_logs)
- Investigating a specific incident in detail (use get_correlated_logs)
- Analyzing changes in log volume over time (use run_log_rate_analysis)`,
    schema: getLogsSchema,
    tags: ['observability', 'logs'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async ({
      index,
      start = DEFAULT_TIME_RANGE.start,
      end = DEFAULT_TIME_RANGE.end,
      terms
    }, {
      esClient
    }) => {
      try {
        const {
          highSeverityCategories,
          lowSeverityCategories
        } = await (0, _handler.getToolHandler)({
          core,
          logger,
          esClient,
          index,
          start,
          end,
          terms
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              highSeverityCategories,
              lowSeverityCategories
            }
          }]
        };
      } catch (error) {
        logger.error(`Error fetching log categories: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch log categories: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}