"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getToolHandler = getToolHandler;
var _get_logs_indices = require("../../utils/get_logs_indices");
var _time = require("../../utils/time");
var _constants = require("./constants");
var _fetch_anchor_logs = require("./fetch_anchor_logs/fetch_anchor_logs");
var _get_correlated_logs_for_anchor = require("./get_correlated_logs_for_anchor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getNoResultsMessage({
  sequences,
  logId,
  logsFilter,
  interestingEventFilter,
  correlationFields,
  start,
  end
}) {
  if (sequences.length > 0) {
    return undefined;
  }
  const isUsingDefaultCorrelationFields = correlationFields === _constants.DEFAULT_CORRELATION_IDENTIFIER_FIELDS;
  const correlationFieldsDescription = isUsingDefaultCorrelationFields ? 'Matching logs exist but lack the default correlation fields (trace.id, request.id, transaction.id, etc.). Try using `correlationFields` for specifying custom correlation fields.' : `Matching logs exist but lack the custom correlation fields: ${correlationFields.join(', ')}`;
  const isUsingDefaultEventFilter = !interestingEventFilter;
  const eventFilterDescription = isUsingDefaultEventFilter ? 'The default `interestingEventFilter` (log.level: ERROR/WARN/FATAL, HTTP 5xx, syslog severity ≤3, etc.) did not match any documents.' : `The 
interestingEventFilter" option "${interestingEventFilter}" did not match any documents.`;
  if (logId) {
    return `The log ID "${logId}" was not found, or the log does not have any of the ${correlationFieldsDescription}.`;
  }
  const suggestions = [`No matching logs exist in this time range (${start} to ${end})`, ...(logsFilter ? ['`logsFilter` is too restrictive'] : []), eventFilterDescription, correlationFieldsDescription];
  return `No log sequences found. Possible reasons: ${suggestions.map((s, i) => `(${i + 1}) ${s}`).join(', ')}.`;
}
async function getToolHandler({
  core,
  logger,
  esClient,
  start,
  end,
  logsFilter,
  interestingEventFilter,
  index,
  correlationFields = _constants.DEFAULT_CORRELATION_IDENTIFIER_FIELDS,
  logId,
  logSourceFields = _constants.DEFAULT_LOG_SOURCE_FIELDS,
  maxSequences = 10,
  maxLogsPerSequence = 200
}) {
  var _index$split;
  const logsIndices = (_index$split = index === null || index === void 0 ? void 0 : index.split(',')) !== null && _index$split !== void 0 ? _index$split : await (0, _get_logs_indices.getLogsIndices)({
    core,
    logger
  });
  const startTime = (0, _time.parseDatemath)(start);
  const endTime = (0, _time.parseDatemath)(end, {
    roundUp: true
  });
  const anchorLogs = await (0, _fetch_anchor_logs.getAnchorLogs)({
    esClient,
    logsIndices,
    startTime,
    endTime,
    logsFilter,
    interestingEventFilter,
    correlationFields,
    logger,
    logId,
    maxSequences
  });

  // For each anchor log, find the correlated logs
  const sequences = await Promise.all(anchorLogs.map(async anchorLog => {
    const {
      logs,
      isTruncated
    } = await (0, _get_correlated_logs_for_anchor.getCorrelatedLogsForAnchor)({
      esClient,
      anchorLog,
      logsIndices,
      logger,
      logSourceFields,
      maxLogsPerSequence
    });
    return {
      correlation: anchorLog.correlation,
      logs,
      isTruncated
    };
  }));
  const message = getNoResultsMessage({
    sequences,
    logId,
    logsFilter,
    interestingEventFilter,
    correlationFields,
    start,
    end
  });
  return {
    sequences,
    message
  };
}