"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchDistributedTrace = fetchDistributedTrace;
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _get_typed_search = require("../../../utils/get_typed_search");
var _dsl_filters = require("../../../utils/dsl_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchDistributedTrace({
  esClient,
  apmIndices,
  traceId,
  start,
  end,
  logger
}) {
  var _traceResponse$aggreg, _traceResponse$aggreg2;
  const search = (0, _get_typed_search.getTypedSearch)(esClient.asCurrentUser);
  const indices = [apmIndices.transaction, apmIndices.span, apmIndices.error].join(',');
  const size = 100;
  const traceResponse = await search({
    index: indices,
    size,
    track_total_hits: size + 1,
    query: {
      bool: {
        filter: [...(0, _dsl_filters.termFilter)('trace.id', traceId), ...(0, _dsl_filters.timeRangeFilter)('@timestamp', {
          start,
          end
        })]
      }
    },
    aggs: {
      services: {
        terms: {
          field: 'service.name',
          size: 100
        },
        aggs: {
          error_count: {
            filter: {
              term: {
                'event.outcome': 'failure'
              }
            }
          }
        }
      }
    },
    fields: ['@timestamp', 'service.name', 'trace.id', 'event.outcome', 'parent.id', 'processor.event', 'transaction.id', 'transaction.name', 'transaction.type', 'transaction.duration.us', 'span.id', 'span.name', 'span.type', 'span.subtype', 'span.duration.us', 'span.destination.service.resource', 'error.id', 'error.grouping_key', 'error.culprit', 'error.log.message', 'error.exception.message', 'error.exception.handled', 'error.exception.type'],
    _source: false,
    sort: [{
      '@timestamp': 'asc'
    }]
  });
  const traceDocuments = traceResponse.hits.hits.map(hit => {
    var _hit$fields;
    return (0, _utils.accessKnownApmEventFields)((_hit$fields = hit.fields) !== null && _hit$fields !== void 0 ? _hit$fields : {}).unflatten();
  });
  const total = traceResponse.hits.total;
  const isPartialTrace = total.relation === 'gte';
  const serviceAggs = (_traceResponse$aggreg = (_traceResponse$aggreg2 = traceResponse.aggregations) === null || _traceResponse$aggreg2 === void 0 ? void 0 : _traceResponse$aggreg2.services.buckets) !== null && _traceResponse$aggreg !== void 0 ? _traceResponse$aggreg : [];
  const services = serviceAggs.map(bucket => {
    var _bucket$error_count$d, _bucket$error_count;
    return {
      serviceName: bucket.key,
      count: bucket.doc_count,
      errorCount: (_bucket$error_count$d = (_bucket$error_count = bucket.error_count) === null || _bucket$error_count === void 0 ? void 0 : _bucket$error_count.doc_count) !== null && _bucket$error_count$d !== void 0 ? _bucket$error_count$d : 0
    };
  }).sort((a, b) => b.count - a.count);
  logger.debug(`Fetched distributed trace for ${traceId}: ${traceDocuments.length} documents, ${services.length} services, partial: ${isPartialTrace}`);
  return {
    traceDocuments,
    isPartialTrace,
    services
  };
}