"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAlertAttachmentType = createAlertAttachmentType;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _dedent = _interopRequireDefault(require("dedent"));
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const alertDataSchema = _zod.z.object({
  alertId: _zod.z.string(),
  attachmentLabel: _zod.z.string().optional()
});
function createAlertAttachmentType({
  core,
  logger
}) {
  return {
    id: _constants.OBSERVABILITY_ALERT_ATTACHMENT_TYPE_ID,
    validate: input => {
      const parsed = alertDataSchema.safeParse(input);
      if (parsed.success) {
        return {
          valid: true,
          data: parsed.data
        };
      }
      return {
        valid: false,
        error: parsed.error.message
      };
    },
    format: attachment => {
      const alertId = attachment.data.alertId;
      return {
        getRepresentation: () => ({
          type: 'text',
          value: `Observability Alert ID: ${alertId}. Use the get_alert_details tool to fetch full alert information.`
        }),
        getBoundedTools: () => [{
          id: `get_alert_details`,
          type: _onechatCommon.ToolType.builtin,
          description: `Fetch full details for alert ${alertId} including rule info, status, reason, and related entities.`,
          schema: _zod.z.object({}),
          handler: async (_, context) => {
            try {
              const [, startDeps] = await core.getStartServices();
              const alertsClient = await startDeps.ruleRegistry.getRacClientWithRequest(context.request);
              const alertDoc = await alertsClient.get({
                id: alertId
              });
              return {
                results: [{
                  type: _tool_result.ToolResultType.other,
                  data: alertDoc
                }]
              };
            } catch (error) {
              logger.error(`Failed to fetch alert ${alertId}: ${error instanceof Error ? error.message : String(error)}`);
              return {
                results: [{
                  type: _tool_result.ToolResultType.error,
                  data: {
                    message: `Failed to fetch alert: ${error instanceof Error ? error.message : String(error)}`
                  }
                }]
              };
            }
          }
        }]
      };
    },
    getTools: () => [],
    getAgentDescription: () => (0, _dedent.default)(`An Observability alert attachment. The alert ID is provided - use the get_alert_details tool to fetch full alert information including rule name, status, reason, and related entities.`)
  };
}