"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDecorationHoveredMessages = exports.filterSuggestionsWithCustomCommands = void 0;
exports.monacoPositionToOffset = monacoPositionToOffset;
exports.offsetRangeToMonacoRange = void 0;
var _lodash = require("lodash");
var _monaco_imports = require("../../../../monaco_imports");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// From Monaco position to linear offset
function monacoPositionToOffset(expression, position) {
  const lines = expression.split(/\n/);
  let offset = 0;
  for (let i = 0; i < position.lineNumber - 1; i++) {
    offset += lines[i].length + 1; // +1 for the newline character
  }

  // one-based to zero-based indexing
  offset += position.column - 1;
  return offset;
}

/**
 * Given an offset range, returns a monaco IRange object.
 *
 * IMPORTANT NOTE:
 * offset ranges are ZERO-based and NOT end-inclusive — [start, end)
 * monaco ranges are ONE-based and ARE end-inclusive — [start, end]
 */
const offsetRangeToMonacoRange = (expression, range) => {
  if (range.start === range.end) {
    return;
  }
  let startColumn = NaN;
  let endColumn = 0;
  let startOfCurrentLine = 0;
  let currentLine = 1;

  // find the line and start column
  for (let i = 0; i < expression.length; i++) {
    if (expression[i] === '\n') {
      currentLine++;
      startOfCurrentLine = i + 1;
    }
    if (i === range.start) {
      startColumn = i + 1 - startOfCurrentLine;
      endColumn = startColumn + range.end - range.start - 1;
      break;
    }
  }
  if (isNaN(startColumn)) {
    return;
  }
  return {
    startLineNumber: currentLine,
    endLineNumber: currentLine,
    startColumn,
    endColumn
  };
};
exports.offsetRangeToMonacoRange = offsetRangeToMonacoRange;
const getDecorationHoveredMessages = (word, position, model) => {
  try {
    const wordRange = new _monaco_imports.monaco.Range(position.lineNumber, word.startColumn, position.lineNumber, word.endColumn);
    const decorations = model.getDecorationsInRange(wordRange);
    return decorations.map(decoration => {
      const hoverMessage = decoration.options.hoverMessage;
      if (!hoverMessage) return '';
      if ((0, _lodash.isArray)(hoverMessage)) {
        return hoverMessage.map(msg => msg.value || '').filter(Boolean).join(', ');
      }
      return hoverMessage.value || '';
    }).filter(Boolean);
  } catch (error) {
    // Silently fail to avoid breaking the hover functionality
    // eslint-disable-next-line no-console
    console.error('Error extracting decoration hover messages:', error);
    return [];
  }
};

/**
 * Extracts the suggestions with custom commands from a list of suggestions.
 * Suggestions with editor.action.triggerSuggest are excluded.
 * @param suggestions
 * @returns
 */
exports.getDecorationHoveredMessages = getDecorationHoveredMessages;
const filterSuggestionsWithCustomCommands = suggestions => {
  return suggestions.filter(suggestion => suggestion.command && suggestion.command.id !== 'editor.action.triggerSuggest').map(suggestion => suggestion.command.id); // we know command is defined because of the filter
};
exports.filterSuggestionsWithCustomCommands = filterSuggestionsWithCustomCommands;