"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildAlertParamsFromAnomaly = buildAlertParamsFromAnomaly;
var _mlAnomalyUtils = require("@kbn/ml-anomaly-utils");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _string_utils = require("../../util/string_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SEVERITY_THRESHOLD_ADJUSTMENT = 5;

/**
 * Builds initial alert parameters from an anomaly record.
 * Pre-populates job ID, severity, result type, and KQL filter based on the anomaly's characteristics.
 */
function buildAlertParamsFromAnomaly(anomaly) {
  const kqlParts = [];

  // Add entity field filters (partition, over, by fields)
  const entityFields = (0, _mlAnomalyUtils.getEntityFieldList)(anomaly.source);
  entityFields.forEach(field => {
    if (field.fieldName && (0, _mlIsDefined.isDefined)(field.fieldValue)) {
      kqlParts.push((0, _string_utils.escapeKueryForFieldValuePair)(field.fieldName, String(field.fieldValue)));
    }
  });
  const actualValue = Array.isArray(anomaly.actual) ? anomaly.actual[0] : anomaly.actual;

  // Add actual value filter if we have a value
  if (typeof actualValue === 'number') {
    // Determine if this is a population job by checking if the anomaly has causes
    // Population jobs have anomalies with a causes array
    const hasCauses = Array.isArray(anomaly.source.causes) && anomaly.source.causes.length > 0;

    // For population jobs (with causes), use nested query syntax: causes: {actual >= value}
    // For regular jobs, use simple comparison: actual >= value
    if (hasCauses) {
      kqlParts.push(`causes: {actual >= ${actualValue}}`);
    } else {
      kqlParts.push(`actual >= ${actualValue}`);
    }
  }

  // Combine all KQL parts
  const kqlQueryString = kqlParts.length > 0 ? kqlParts.join(' and ') : null;

  // Set severity slightly lower than the anomaly to catch similar anomalies
  const severity = Math.max(0, Math.floor(anomaly.severity) - SEVERITY_THRESHOLD_ADJUSTMENT);
  return {
    jobSelection: {
      jobIds: [anomaly.jobId]
    },
    severity,
    resultType: _mlAnomalyUtils.ML_ANOMALY_RESULT_TYPE.RECORD,
    includeInterim: false,
    kqlQueryString
  };
}