"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.searchFieldsRoute = exports.getFieldsRoute = exports.fieldsRoutes = void 0;
var _zod = require("@kbn/zod");
var _tracedEsClient = require("@kbn/traced-es-client");
var _zodHelpers = require("@kbn/zod-helpers");
var _datemath = require("@kbn/datemath");
var _server = require("@kbn/data-plugin/server");
var _get_metric_fields = require("./get_metric_fields");
var _create_route = require("../create_route");
var _utils = require("../../lib/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getFieldsRoute = exports.getFieldsRoute = (0, _create_route.createRoute)({
  endpoint: 'GET /internal/metrics_experience/fields',
  security: {
    authz: {
      enabled: false,
      reason: 'Authorization provided by Elasticsearch'
    }
  },
  params: _zod.z.object({
    query: _zod.z.object({
      index: _zod.z.string().default('metrics-*'),
      to: _zod.z.string().datetime().default((0, _datemath.parse)('now').toISOString()).transform(_zodHelpers.isoToEpoch),
      from: _zod.z.string().datetime().default((0, _datemath.parse)('now-15m', {
        roundUp: true
      }).toISOString()).transform(_zodHelpers.isoToEpoch),
      fields: _zod.z.union([_zod.z.string(), _zod.z.array(_zod.z.string())]).default('*'),
      page: _zod.z.coerce.number().int().positive().default(1),
      size: _zod.z.coerce.number().int().positive().default(100)
    })
  }),
  handler: async ({
    context,
    params,
    logger,
    request
  }) => {
    const {
      elasticsearch,
      featureFlags
    } = await context.core;
    await (0, _utils.throwNotFoundIfMetricsExperienceDisabled)(featureFlags);
    const esClient = elasticsearch.client.asCurrentUser;
    const page = params.query.page;
    const size = params.query.size;
    const {
      fields,
      total
    } = await (0, _get_metric_fields.getMetricFields)({
      esClient: (0, _tracedEsClient.createTracedEsClient)({
        logger,
        client: esClient,
        plugin: 'metrics_experience',
        abortSignal: (0, _server.getRequestAbortedSignal)(request.events.aborted$)
      }),
      indexPattern: params.query.index,
      timerange: {
        from: params.query.from,
        to: params.query.to
      },
      fields: params.query.fields,
      page,
      size,
      logger
    });
    return {
      fields,
      total,
      page
    };
  }
});
const searchFieldsRoute = exports.searchFieldsRoute = (0, _create_route.createRoute)({
  endpoint: 'POST /internal/metrics_experience/fields/_search',
  security: {
    authz: {
      requiredPrivileges: ['read']
    }
  },
  params: _zod.z.object({
    body: _zod.z.object({
      index: _zod.z.string().default('metrics-*'),
      to: _zod.z.string().datetime().default((0, _datemath.parse)('now').toISOString()).transform(_zodHelpers.isoToEpoch),
      from: _zod.z.string().datetime().default((0, _datemath.parse)('now-15m', {
        roundUp: true
      }).toISOString()).transform(_zodHelpers.isoToEpoch),
      fields: _zod.z.union([_zod.z.string(), _zod.z.array(_zod.z.string())]).default('*'),
      page: _zod.z.coerce.number().int().positive().default(1),
      size: _zod.z.coerce.number().int().positive().default(100),
      filters: _zod.z.record(_zod.z.string(), _zod.z.array(_zod.z.string()).max(100)).optional().default({})
    })
  }),
  handler: async ({
    context,
    params,
    logger
  }) => {
    const {
      elasticsearch,
      featureFlags
    } = await context.core;
    await (0, _utils.throwNotFoundIfMetricsExperienceDisabled)(featureFlags);
    const esClient = elasticsearch.client.asCurrentUser;
    const {
      index,
      from,
      to,
      fields,
      page,
      size,
      filters
    } = params.body;
    const {
      fields: resultFields,
      total
    } = await (0, _get_metric_fields.getMetricFields)({
      esClient: (0, _tracedEsClient.createTracedEsClient)({
        logger,
        client: esClient,
        plugin: 'metrics_experience'
      }),
      indexPattern: index,
      timerange: {
        from,
        to
      },
      fields: fields === '*' ? fields : (Array.isArray(fields) ? fields : [fields]).concat(Object.keys(filters)),
      page,
      size,
      filters,
      logger
    });
    return {
      fields: resultFields,
      total,
      page
    };
  }
});
const fieldsRoutes = exports.fieldsRoutes = {
  ...getFieldsRoute,
  ...searchFieldsRoute
};