"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.McpClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _auth = require("@modelcontextprotocol/sdk/client/auth.js");
var _index = require("@modelcontextprotocol/sdk/client/index.js");
var _streamableHttp = require("@modelcontextprotocol/sdk/client/streamableHttp.js");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * McpClient is a wrapper around the MCP client SDK.
 * It provides a simple interface for connecting to an MCP client,
 * listing tools, and calling tools.
 */
class McpClient {
  constructor(logger, clientDetails, {
    headers = {},
    maxRetries = 3,
    reconnectionDelayGrowFactor = 1.5,
    initialReconnectionDelay = 1000,
    maxReconnectionDelay = 10000
  } = {}) {
    (0, _defineProperty2.default)(this, "client", void 0);
    (0, _defineProperty2.default)(this, "transport", void 0);
    (0, _defineProperty2.default)(this, "connected", false);
    (0, _defineProperty2.default)(this, "name", void 0);
    (0, _defineProperty2.default)(this, "version", void 0);
    this.logger = logger;
    this.transport = new _streamableHttp.StreamableHTTPClientTransport(new URL(clientDetails.url), {
      requestInit: {
        headers
      },
      reconnectionOptions: {
        maxRetries,
        reconnectionDelayGrowFactor,
        initialReconnectionDelay,
        maxReconnectionDelay
      }
    });
    this.name = clientDetails.name;
    this.version = clientDetails.version;
    this.client = new _index.Client({
      name: clientDetails.name,
      version: clientDetails.version
    });
  }

  /**
   * Public getter for the connection status.
   */
  isConnected() {
    return this.connected;
  }

  /**
   * Connect to the MCP client and return the connected status and capabilities.
   */
  async connect() {
    if (!this.connected) {
      this.logger.info(`Attempting to connect to MCP server ${this.name}, ${this.version}`);
      try {
        // connect() performs the initialization handshake with the MCP server as per MCP protocol
        await this.client.connect(this.transport);
        this.connected = true;
        this.logger.info(`Connected to MCP server ${this.name}, ${this.version}`);
      } catch (error) {
        const message = error instanceof Error ? error.message : String(error);
        this.logger.error(`Error connecting to MCP server ${this.name}, ${this.version}: ${message}`);
        if (error instanceof _streamableHttp.StreamableHTTPError) {
          // The SDK formats the message as "Streamable HTTP error: Connection failed"
          throw new Error(`${message}`);
        } else if (error instanceof _auth.UnauthorizedError) {
          throw new Error(`Unauthorized error: ${message}`);
        } else {
          throw new Error(`Error connecting to MCP server: ${message}`);
        }
      }
    }
    // return the full list of capabilities as a by-product of the initialization handshake
    const capabilities = this.client.getServerCapabilities();
    return {
      connected: this.connected,
      capabilities
    };
  }

  /**
   * Disconnect from the MCP client and return the disconnected status.
   */
  async disconnect() {
    if (this.connected) {
      this.logger.info(`Attempting to disconnect from MCP server ${this.name}, ${this.version}`);
      await this.client.close();
      this.connected = false;
      this.logger.info(`Disconnected from MCP client ${this.name}, ${this.version}`);
    }
  }

  /**
   * List the tools available on the MCP client.
   */
  async listTools() {
    if (!this.connected) {
      throw new Error(`MCP client not connected to ${this.name}, ${this.version}`);
    }
    this.logger.info(`Listing tools from MCP server ${this.name}, ${this.version}`);
    const getNextPage = async cursor => {
      const response = await this.client.listTools({
        cursor
      });
      if (response.isError) {
        throw new Error(`Error listing tools: ${response.error}`);
      }
      const {
        tools,
        nextCursor
      } = response;
      return [...tools.map(tool => {
        return {
          description: tool.description,
          inputSchema: tool.inputSchema,
          name: tool.name
        };
      }), ...(nextCursor ? await getNextPage(nextCursor) : [])];
    };
    const tools = {
      tools: await getNextPage()
    };
    return tools;
  }

  /**
   * Call a tool on the MCP client.
   * This method only returns text content.
   * It does not support other content types such as images, audio, etc.
   * @param {CallToolParams} params - The parameters for the tool call.
   */
  async callTool(params) {
    if (!this.connected) {
      throw new Error(`MCP client not connected to ${this.name}, ${this.version}`);
    }
    this.logger.info(`Calling tool ${params.name} on MCP server ${this.name}, ${this.version}`);
    const response = await this.client.callTool({
      name: params.name,
      arguments: params.arguments
    });
    if (response.isError) {
      throw new Error(`Error calling tool ${params.name} with ${params.arguments}: ${response.error}`);
    }
    const content = response.content;
    const textParts = content.filter(_types.isTextPart);
    return {
      content: textParts
    };
  }
}
exports.McpClient = McpClient;