"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SCHEDULE = exports.MAINTENANCE_WINDOW_EVENTS_TASK_TYPE = exports.MAINTENANCE_WINDOW_EVENTS_TASK_ID = void 0;
exports.createEventsGeneratorTaskRunner = createEventsGeneratorTaskRunner;
exports.generateEvents = generateEvents;
exports.getSOFinder = getSOFinder;
exports.getStatusFilter = getStatusFilter;
exports.initializeMaintenanceWindowEventsGenerator = initializeMaintenanceWindowEventsGenerator;
exports.scheduleMaintenanceWindowEventsGenerator = scheduleMaintenanceWindowEventsGenerator;
exports.updateMaintenanceWindowsEvents = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _esQuery = require("@kbn/es-query");
var _common = require("../../common");
var _transforms = require("../application/transforms");
var _get_maintenance_window_status = require("../application/lib/get_maintenance_window_status");
var _generate_maintenance_window_events = require("../application/lib/generate_maintenance_window_events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAINTENANCE_WINDOW_EVENTS_TASK_TYPE = exports.MAINTENANCE_WINDOW_EVENTS_TASK_TYPE = 'maintenance-window:generate-events';
const MAINTENANCE_WINDOW_EVENTS_TASK_ID = exports.MAINTENANCE_WINDOW_EVENTS_TASK_ID = `${MAINTENANCE_WINDOW_EVENTS_TASK_TYPE}-generator`;
const SCHEDULE = exports.SCHEDULE = {
  interval: '1d'
};
function initializeMaintenanceWindowEventsGenerator(logger, taskManager, coreStartServices) {
  registerMaintenanceWindowEventsGeneratorTask(logger, taskManager, coreStartServices);
}
async function scheduleMaintenanceWindowEventsGenerator(logger, taskManager) {
  try {
    await taskManager.ensureScheduled({
      id: MAINTENANCE_WINDOW_EVENTS_TASK_ID,
      taskType: MAINTENANCE_WINDOW_EVENTS_TASK_TYPE,
      schedule: SCHEDULE,
      state: {},
      params: {}
    });
  } catch (e) {
    logger.error(`Error scheduling ${MAINTENANCE_WINDOW_EVENTS_TASK_ID}, received ${e.message}`);
  }
}
function registerMaintenanceWindowEventsGeneratorTask(logger, taskManager, coreStartServices) {
  taskManager.registerTaskDefinitions({
    [MAINTENANCE_WINDOW_EVENTS_TASK_TYPE]: {
      title: 'Maintenance window events generator task',
      createTaskRunner: createEventsGeneratorTaskRunner(logger, coreStartServices),
      timeout: '30m'
    }
  });
}
function createEventsGeneratorTaskRunner(logger, coreStartServices) {
  return () => {
    let cancelled = false;
    let soFinder;
    return {
      async run() {
        try {
          const [{
            savedObjects
          }] = await coreStartServices();
          const savedObjectsClient = savedObjects.createInternalRepository([_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE]);

          // we are using total 2 weeks range to find maintenance windows that are expiring
          const startRangeDate = (0, _moment.default)().startOf('day').utc().subtract(1, 'week').toISOString(); // 1 week before current date
          const endRangeDate = (0, _moment.default)().startOf('day').utc().add(1, 'week').toISOString(); // 1 week after current date

          const startRangeFilter = _esQuery.nodeBuilder.range('maintenance-window.attributes.expirationDate', 'gte', startRangeDate);
          const endRangeFilter = _esQuery.nodeBuilder.range('maintenance-window.attributes.expirationDate', 'lte', endRangeDate);
          const statusFilter = getStatusFilter();
          const filter = _esQuery.nodeBuilder.and([startRangeFilter, endRangeFilter, statusFilter]);
          soFinder = getSOFinder({
            savedObjectsClient,
            logger,
            filter
          });
          const totalMaintenanceWindowsWithGeneratedEvents = await updateMaintenanceWindowsEvents({
            savedObjectsClient,
            logger,
            startRangeDate,
            soFinder
          });
          logger.debug(`Maintenance windows events generator task updated ${totalMaintenanceWindowsWithGeneratedEvents} maintenance windows successfully`);
        } catch (e) {
          logger.warn(`Error executing maintenance windows events generator task: ${e.message}`);
        }
      },
      async cancel() {
        var _soFinder;
        if (cancelled) {
          return;
        }
        logger.debug(`Cancelling maintenance windows events generator task - execution error due to timeout.`);
        cancelled = true;
        await ((_soFinder = soFinder) === null || _soFinder === void 0 ? void 0 : _soFinder.close());
        return;
      }
    };
  };
}
function getStatusFilter() {
  const mwStatusQuery = (0, _get_maintenance_window_status.getMaintenanceWindowStatus)();
  const fullQuery = [_common.MaintenanceWindowStatus.Running, _common.MaintenanceWindowStatus.Upcoming, _common.MaintenanceWindowStatus.Finished].map(value => mwStatusQuery[value]).filter(Boolean).join(' or ');
  return (0, _esQuery.fromKueryExpression)(fullQuery);
}
const updateMaintenanceWindowsEvents = async ({
  soFinder,
  savedObjectsClient,
  logger,
  startRangeDate
}) => {
  let totalUpdatedMaintenanceWindows = 0;
  let mwsWithNewEvents = [];
  let mwSOWithErrors = 0;
  if (soFinder) {
    for await (const findResults of soFinder.find()) {
      try {
        mwsWithNewEvents = await generateEvents({
          maintenanceWindowsSO: findResults.saved_objects,
          startRangeDate
        });
        if (mwsWithNewEvents.length) {
          const bulkUpdateReq = mwsWithNewEvents.map(mw => {
            const updatedMaintenanceWindowAttributes = (0, _transforms.transformMaintenanceWindowToMaintenanceWindowAttributes)({
              ...mw
            });
            return {
              type: _common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE,
              id: mw.id,
              attributes: updatedMaintenanceWindowAttributes
            };
          });
          const result = await savedObjectsClient.bulkUpdate(bulkUpdateReq);
          for (const savedObject of result.saved_objects) {
            if (savedObject.error) {
              logger.error(`MW event generator: Failed to update maintenance window "${savedObject.id}". Error: ${savedObject.error.message}`);
              mwSOWithErrors++;
            }
          }
          totalUpdatedMaintenanceWindows = totalUpdatedMaintenanceWindows + (result.saved_objects.length - mwSOWithErrors);
        }
      } catch (e) {
        logger.error(`MW event generator: Failed to update events in maintenance windows saved object". Error: ${e.message}`);
      }
    }
    await soFinder.close();
  }
  logger.debug(`Total updated maintenance windows "${totalUpdatedMaintenanceWindows}"`);
  return totalUpdatedMaintenanceWindows;
};
exports.updateMaintenanceWindowsEvents = updateMaintenanceWindowsEvents;
function getSOFinder({
  savedObjectsClient,
  logger,
  filter
}) {
  try {
    return savedObjectsClient.createPointInTimeFinder({
      type: _common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE,
      namespaces: ['*'],
      perPage: 1000,
      sortField: 'updatedAt',
      sortOrder: 'desc',
      filter
    });
  } catch (e) {
    logger.error(`MW event generator: Failed instantiate a createPointInTimeFinder instance". Error: ${e.message}`);
    return null;
  }
}
async function generateEvents({
  maintenanceWindowsSO,
  startRangeDate
}) {
  const maintenanceWindows = maintenanceWindowsSO.map(savedObject => (0, _transforms.transformMaintenanceWindowAttributesToMaintenanceWindow)({
    attributes: savedObject.attributes,
    id: savedObject.id
  }));

  // 1 year from the task run date (current date) till the end of the day
  const newExpirationDate = (0, _moment.default)().utc().endOf('day').add(1, 'year').toISOString();
  try {
    const mwWithGeneratedEvents = maintenanceWindows
    // filtering the maintenance windows that have recurring schedule and events
    .filter(maintenanceWindow => (maintenanceWindow.rRule.interval !== undefined || maintenanceWindow.rRule.freq !== undefined) && maintenanceWindow.events.length).map(filteredMaintenanceWindow => {
      const {
        rRule,
        duration,
        expirationDate: oldExpirationDate
      } = filteredMaintenanceWindow;
      const newEvents = (0, _generate_maintenance_window_events.generateMaintenanceWindowEvents)({
        rRule,
        expirationDate: newExpirationDate,
        duration,
        startDate: startRangeDate // here start range date is 1 week before current expiration date
      });
      return {
        ...filteredMaintenanceWindow,
        expirationDate: newEvents.length ? newExpirationDate : oldExpirationDate,
        events: [...newEvents]
      };
    }).filter(mappedMW => mappedMW.expirationDate === newExpirationDate);
    return mwWithGeneratedEvents;
  } catch (e) {
    throw new Error(`Failed to generate events for maintenance windows. Error: ${e.message}`);
  }
}