"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteListIndexRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _api = require("@kbn/securitysolution-lists-common/api");
var _constants = require("@kbn/security-solution-features/constants");
var _utils = require("../utils");
var _2 = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Deletes all of the indexes, template, ilm policies, and aliases. You can check
 * this by looking at each of these settings from ES after a deletion:
 *
 * GET /_template/.lists-default
 * GET /.lists-default-000001/
 * GET /_ilm/policy/.lists-default
 * GET /_alias/.lists-default
 *
 * GET /_template/.items-default
 * GET /.items-default-000001/
 * GET /_ilm/policy/.items-default
 * GET /_alias/.items-default
 *
 * And ensuring they're all gone
 */
const deleteListIndexRoute = router => {
  router.versioned.delete({
    access: 'public',
    path: _securitysolutionListConstants.LIST_INDEX,
    security: {
      authz: {
        requiredPrivileges: [_constants.LISTS_API_ALL]
      }
    }
  }).addVersion({
    validate: false,
    version: '2023-10-31'
  }, async (context, _, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const lists = await (0, _2.getListClient)(context);
      const listIndexExists = await lists.getListIndexExists();
      const listItemIndexExists = await lists.getListItemIndexExists();
      const listDataStreamExists = await lists.getListDataStreamExists();
      const listItemDataStreamExists = await lists.getListItemDataStreamExists();

      // return early if no data stream or indices exist
      if (!listDataStreamExists && !listItemDataStreamExists && !listIndexExists && !listItemIndexExists) {
        return siemResponse.error({
          body: `index and data stream: "${lists.getListName()}" and "${lists.getListItemName()}" does not exist`,
          statusCode: 404
        });
      }

      // ensure data streams deleted if exist
      await deleteDataStreams(lists, listDataStreamExists, listItemDataStreamExists);

      // we need to call this section only if any of these indices exist
      // to delete indices, ILM policies and legacy templates
      // ILM policies and legacy templates do not exist on serverless,
      // so by checking if any of index exists we ensure it is stateful
      if (listIndexExists || listItemIndexExists) {
        await deleteIndices(lists, listIndexExists, listItemIndexExists);
        await lists.deleteLegacyListTemplateIfExists();
        await lists.deleteLegacyListItemTemplateIfExists();
      }
      await deleteIndexTemplates(lists);
      return response.ok({
        body: _api.DeleteListIndexResponse.parse({
          acknowledged: true
        })
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};

/**
 * Delete list/item indices
 */
exports.deleteListIndexRoute = deleteListIndexRoute;
const deleteIndices = async (lists, listIndexExists, listItemIndexExists) => {
  if (listIndexExists) {
    await lists.deleteListIndex();
  }
  if (listItemIndexExists) {
    await lists.deleteListItemIndex();
  }
  const listsPolicyExists = await lists.getListPolicyExists();
  const listItemPolicyExists = await lists.getListItemPolicyExists();
  if (listsPolicyExists) {
    await lists.deleteListPolicy();
  }
  if (listItemPolicyExists) {
    await lists.deleteListItemPolicy();
  }
};

/**
 * Delete list/item data streams
 */
const deleteDataStreams = async (lists, listDataStreamExists, listItemDataStreamExists) => {
  if (listDataStreamExists) {
    await lists.deleteListDataStream();
  }
  if (listItemDataStreamExists) {
    await lists.deleteListItemDataStream();
  }
};

/**
 * Delete list/item index templates
 */
const deleteIndexTemplates = async lists => {
  const listsTemplateExists = await lists.getListTemplateExists();
  const listItemTemplateExists = await lists.getListItemTemplateExists();
  if (listsTemplateExists) {
    await lists.deleteListTemplate();
  }
  if (listItemTemplateExists) {
    await lists.deleteListItemTemplate();
  }
};