"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLicenseFetcher = void 0;
var _crypto = require("crypto");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _jsonStableStringify = _interopRequireDefault(require("json-stable-stringify"));
var _std = require("@kbn/std");
var _license = require("../common/license");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getLicenseFetcher = ({
  clusterClient,
  logger,
  cacheDurationMs,
  maxRetryDelay
}) => {
  let currentLicense;
  let lastSuccessfulFetchTime;
  const maxRetries = Math.floor(Math.log2(maxRetryDelay / 1000)) + 1;
  return async () => {
    const client = (0, _std.isPromise)(clusterClient) ? await clusterClient : clusterClient;
    try {
      const response = await (0, _pRetry.default)(() => client.asInternalUser.xpack.info(), {
        retries: maxRetries
      });
      const normalizedLicense = response.license && response.license.type !== 'missing' ? normalizeServerLicense(response.license) : undefined;
      const normalizedFeatures = response.features ? normalizeFeatures(response.features) : undefined;
      if (!normalizedLicense) {
        logger.warn('License information fetched from Elasticsearch, but no license is available');
      }
      const signature = sign({
        license: normalizedLicense,
        features: normalizedFeatures,
        error: ''
      });
      currentLicense = new _license.License({
        license: normalizedLicense,
        features: normalizedFeatures,
        signature
      });
      lastSuccessfulFetchTime = Date.now();
      return currentLicense;
    } catch (err) {
      var _err$originalError;
      const error = (_err$originalError = err.originalError) !== null && _err$originalError !== void 0 ? _err$originalError : err;
      logger.warn(`License information could not be obtained from Elasticsearch due to ${error} error`);
      if (lastSuccessfulFetchTime && lastSuccessfulFetchTime + cacheDurationMs > Date.now()) {
        return currentLicense;
      } else {
        const errorMessage = getErrorMessage(error);
        const signature = sign({
          error: errorMessage
        });
        return new _license.License({
          error: getErrorMessage(error),
          signature
        });
      }
    }
  };
};
exports.getLicenseFetcher = getLicenseFetcher;
function normalizeServerLicense(license) {
  return {
    uid: license.uid,
    type: license.type,
    mode: license.mode,
    expiryDateInMillis: typeof license.expiry_date_in_millis === 'string' ? parseInt(license.expiry_date_in_millis, 10) : license.expiry_date_in_millis,
    status: license.status
  };
}
function normalizeFeatures(rawFeatures) {
  const features = {};
  for (const [id, feature] of Object.entries(rawFeatures)) {
    features[id] = {
      isAvailable: feature.available,
      isEnabled: feature.enabled
    };
  }
  return features;
}
function sign({
  license,
  features,
  error
}) {
  return (0, _crypto.createHash)('sha256').update((0, _jsonStableStringify.default)({
    license,
    features,
    error
  })).digest('hex');
}
function getErrorMessage(error) {
  if (error.status === 400) {
    return 'X-Pack plugin is not installed on the Elasticsearch cluster.';
  }
  return error.message;
}