"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestionsApi = void 0;
var _visualizationUtils = require("@kbn/visualization-utils");
var _suggestion_helpers = require("../editor_frame_service/editor_frame/suggestion_helpers");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Helper function to find compatible suggestion by chart type
const findCompatibleSuggestion = (suggestionCandidates, targetChartType) => suggestionCandidates.find(s => s.title.includes(targetChartType) || s.visualizationId.includes(targetChartType));

// Helper function to merge suggestion with visual attributes if needed
const createSuggestionWithAttributes = (suggestion, preferredVisAttributes, context) => preferredVisAttributes ? (0, _helpers.mergeSuggestionWithVisContext)({
  suggestion,
  visAttributes: preferredVisAttributes,
  context
}) : suggestion;
const suggestionsApi = ({
  context,
  dataView,
  datasourceMap,
  visualizationMap,
  excludedVisualizations,
  preferredChartType,
  preferredVisAttributes
}) => {
  var _initialVisualization;
  const initialContext = context;
  if (!datasourceMap || !visualizationMap || !dataView.id) return undefined;
  const datasourceStates = {
    formBased: {
      isLoading: false,
      state: {
        layers: {}
      }
    },
    textBased: {
      isLoading: false,
      state: {
        layers: {},
        indexPatternRefs: [],
        initialContext
      }
    }
  };
  const currentDataViewId = dataView.id;
  const dataViews = {
    indexPatterns: {
      [currentDataViewId]: dataView
    },
    indexPatternRefs: []
  };
  const initialVisualization = (visualizationMap === null || visualizationMap === void 0 ? void 0 : visualizationMap[Object.keys(visualizationMap)[0]]) || null;
  const isInitialSubTypeSupported = preferredChartType ? initialVisualization === null || initialVisualization === void 0 ? void 0 : (_initialVisualization = initialVisualization.isSubtypeSupported) === null || _initialVisualization === void 0 ? void 0 : _initialVisualization.call(initialVisualization, preferredChartType.toLowerCase()) : undefined;

  // find the active visualizations from the context
  const suggestions = (0, _suggestion_helpers.getSuggestions)({
    datasourceMap,
    datasourceStates,
    visualizationMap,
    activeVisualization: initialVisualization,
    visualizationState: undefined,
    visualizeTriggerFieldContext: context,
    subVisualizationId: isInitialSubTypeSupported ? preferredChartType === null || preferredChartType === void 0 ? void 0 : preferredChartType.toLowerCase() : undefined,
    dataViews
  });
  if (!suggestions.length) return [];
  const primarySuggestion = suggestions[0];
  const activeVisualization = visualizationMap[primarySuggestion.visualizationId];
  if (primarySuggestion.incomplete || excludedVisualizations !== null && excludedVisualizations !== void 0 && excludedVisualizations.includes(primarySuggestion.visualizationId)) {
    return [];
  }
  // compute the rest suggestions depending on the active one and filter out the lnsLegacyMetric
  const newSuggestions = (0, _suggestion_helpers.getSuggestions)({
    datasourceMap,
    datasourceStates: {
      textBased: {
        isLoading: false,
        state: primarySuggestion.datasourceState
      }
    },
    visualizationMap,
    activeVisualization,
    visualizationState: primarySuggestion.visualizationState,
    dataViews
  }).filter(sug =>
  // Datatables are always return as hidden suggestions
  // if the user has requested for a datatable (preferredChartType), we want to return it
  // although it is a hidden suggestion
  sug.hide && sug.visualizationId === 'lnsDatatable' ||
  // Filter out suggestions that are hidden and legacy metrics
  !sug.hide && sug.visualizationId !== 'lnsLegacyMetric');
  const chartType = preferredChartType === null || preferredChartType === void 0 ? void 0 : preferredChartType.toLowerCase();

  // to return line / area instead of a bar chart
  const xyResult = (0, _helpers.switchVisualizationType)({
    visualizationMap,
    suggestions: newSuggestions,
    targetTypeId: chartType,
    familyType: 'lnsXY',
    forceSwitch: ['area', 'line'].some(type => chartType === null || chartType === void 0 ? void 0 : chartType.includes(type))
  });
  if (xyResult) return xyResult;

  // to return a donut instead of a pie chart
  const pieResult = (0, _helpers.switchVisualizationType)({
    visualizationMap,
    suggestions: newSuggestions,
    targetTypeId: chartType,
    familyType: 'lnsPie',
    forceSwitch: preferredChartType === _visualizationUtils.ChartType.Donut
  });
  if (pieResult) return pieResult;
  const chartTypeFromAttrs = preferredVisAttributes ? (0, _visualizationUtils.mapVisToChartType)(preferredVisAttributes.visualizationType) : undefined;
  const targetChartType = preferredChartType !== null && preferredChartType !== void 0 ? preferredChartType : chartTypeFromAttrs;

  // in case the user asks for another type (except from area, line) check if it exists
  // in suggestions and return this instead
  const suggestionsList = [primarySuggestion, ...newSuggestions];

  // Handle preferred chart type logic
  if (targetChartType) {
    // Special case for table when user hasn't changed chart type and there's only one suggestion
    if (!preferredChartType && suggestionsList.length === 1 && targetChartType === _visualizationUtils.ChartType.Table) {
      const suggestion = createSuggestionWithAttributes(suggestionsList[0], preferredVisAttributes, context);
      return [suggestion];
    }

    // General case: find compatible suggestion for preferred chart type
    // Skip if user hasn't changed chart type, has multiple suggestions, and wants table
    const shouldSkipSearch = !preferredChartType && suggestionsList.length > 1 && targetChartType === _visualizationUtils.ChartType.Table;
    if (!shouldSkipSearch) {
      const compatibleSuggestion = findCompatibleSuggestion(suggestionsList, targetChartType);
      const selectedSuggestion = compatibleSuggestion !== null && compatibleSuggestion !== void 0 ? compatibleSuggestion : suggestionsList[0];
      return [createSuggestionWithAttributes(selectedSuggestion, preferredVisAttributes, context)];
    }
  }

  // if there is no preference from the user, send everything
  // until we separate the text based suggestions logic from the dataview one,
  // we want to sort XY first
  const sortXYFirst = suggestionsList.sort((a, b) => a.visualizationId === 'lnsXY' ? -1 : 1);
  return sortXYFirst;
};
exports.suggestionsApi = suggestionsApi;