"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMissingIndexPattern = getMissingIndexPattern;
exports.initializeDataViews = initializeDataViews;
exports.initializeDatasources = initializeDatasources;
exports.initializeSources = initializeSources;
exports.initializeVisualization = initializeVisualization;
exports.persistedStateToExpression = persistedStateToExpression;
var _lodash = require("lodash");
var _coloring = require("@kbn/coloring");
var _common = require("@kbn/data-views-plugin/common");
var _eventAnnotationCommon = require("@kbn/event-annotation-common");
var _constants = require("../../../common/constants");
var _expression_helpers = require("./expression_helpers");
var _utils = require("../../utils");
var _settings_storage = require("../../settings_storage");
var _loader = require("../../data_views_service/loader");
var _utils2 = require("../../state_management/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// there are 2 ways of coloring, the color mapping where the user can map specific colors to
// specific terms, and the palette assignment where the colors are assinged automatically
// by a palette with rotating the colors
const COLORING_METHOD = _constants.COLOR_MAPPING_OFF_BY_DEFAULT ? {
  type: 'legacyPalette',
  value: {
    name: 'default',
    type: 'palette'
  }
} : {
  type: 'colorMapping',
  value: {
    ..._coloring.DEFAULT_COLOR_MAPPING_CONFIG
  }
};
function getIndexPatterns(annotationGroupDataviewIds, references, initialContext, initialId, adHocDataviews) {
  const indexPatternIds = [...annotationGroupDataviewIds];

  // use the initialId only when no context is passed over
  if (!initialContext && initialId) {
    indexPatternIds.push(initialId);
  }
  if (initialContext) {
    if ('isVisualizeAction' in initialContext) {
      indexPatternIds.push(...initialContext.indexPatternIds);
    } else {
      indexPatternIds.push(initialContext.dataViewSpec.id);
    }
  }
  if (references) {
    for (const reference of references) {
      if (reference.type === 'index-pattern') {
        indexPatternIds.push(reference.id);
      }
    }
  }
  if (adHocDataviews) {
    indexPatternIds.push(...adHocDataviews);
  }
  return [...new Set(indexPatternIds)];
}
const getLastUsedIndexPatternId = (storage, indexPatternRefs) => {
  var _indexPatternRefs$fin;
  const indexPattern = (0, _settings_storage.readFromStorage)(storage, 'indexPatternId');
  return indexPattern && ((_indexPatternRefs$fin = indexPatternRefs.find(i => i.id === indexPattern)) === null || _indexPatternRefs$fin === void 0 ? void 0 : _indexPatternRefs$fin.id);
};
const getRefsForAdHocDataViewsFromContext = (indexPatternRefs, usedIndexPatternsIds, indexPatterns) => {
  const indexPatternIds = indexPatternRefs.map(({
    id
  }) => id);
  const adHocDataViewsIds = usedIndexPatternsIds.filter(id => !indexPatternIds.includes(id));
  const adHocDataViewsList = Object.values(indexPatterns).filter(({
    id
  }) => adHocDataViewsIds.includes(id));
  return adHocDataViewsList.map(({
    id,
    title,
    name
  }) => ({
    id,
    title,
    name
  }));
};
async function initializeDataViews({
  dataViews,
  datasourceMap,
  datasourceStates,
  storage,
  defaultIndexPatternId,
  references,
  initialContext,
  adHocDataViews: persistedAdHocDataViews,
  annotationGroups
}, options) {
  var _indexPatternRefs$;
  const adHocDataViews = Object.fromEntries(Object.entries(persistedAdHocDataViews || {}).map(([id, persistedSpec]) => {
    const spec = _common.DataViewPersistableStateService.inject(persistedSpec, references || []);
    return [id, spec];
  }));
  const annotationGroupValues = Object.values(annotationGroups);
  for (const group of annotationGroupValues) {
    var _group$dataViewSpec;
    if ((_group$dataViewSpec = group.dataViewSpec) !== null && _group$dataViewSpec !== void 0 && _group$dataViewSpec.id) {
      adHocDataViews[group.dataViewSpec.id] = group.dataViewSpec;
    }
  }
  const {
    isFullEditor
  } = options !== null && options !== void 0 ? options : {};

  // make it explicit or TS will infer never[] and break few lines down
  const indexPatternRefs = await (isFullEditor ? (0, _loader.loadIndexPatternRefs)(dataViews) : []);

  // if no state is available, use the fallbackId
  const lastUsedIndexPatternId = getLastUsedIndexPatternId(storage, indexPatternRefs);
  const fallbackId = lastUsedIndexPatternId || defaultIndexPatternId || ((_indexPatternRefs$ = indexPatternRefs[0]) === null || _indexPatternRefs$ === void 0 ? void 0 : _indexPatternRefs$.id);
  const initialId = !initialContext && Object.keys(datasourceMap).every(datasourceId => {
    var _datasourceStates$dat;
    return !((_datasourceStates$dat = datasourceStates[datasourceId]) !== null && _datasourceStates$dat !== void 0 && _datasourceStates$dat.state);
  }) ? fallbackId : undefined;
  const adHocDataviewsIds = Object.keys(adHocDataViews || {});
  const usedIndexPatternsIds = getIndexPatterns(annotationGroupValues.map(group => group.indexPatternId), references, initialContext, initialId, adHocDataviewsIds);

  // load them
  const availableIndexPatterns = new Set(indexPatternRefs.map(({
    id
  }) => id));
  const notUsedPatterns = (0, _lodash.difference)([...availableIndexPatterns], usedIndexPatternsIds);
  const indexPatterns = await (0, _loader.loadIndexPatterns)({
    dataViews,
    patterns: usedIndexPatternsIds,
    notUsedPatterns,
    cache: {},
    adHocDataViews
  });
  const adHocDataViewsRefs = getRefsForAdHocDataViewsFromContext(indexPatternRefs, usedIndexPatternsIds, indexPatterns);
  return {
    indexPatternRefs: (0, _utils.sortDataViewRefs)([...indexPatternRefs, ...adHocDataViewsRefs]),
    indexPatterns
  };
}
const initializeEventAnnotationGroups = async (eventAnnotationService, references) => {
  const annotationGroups = {};
  await Promise.allSettled((references || []).filter(ref => ref.type === _eventAnnotationCommon.EVENT_ANNOTATION_GROUP_TYPE).map(({
    id
  }) => eventAnnotationService.loadAnnotationGroup(id).then(group => {
    annotationGroups[id] = group;
  })));
  return annotationGroups;
};

/**
 * This function composes both initializeDataViews & initializeDatasources into a single call
 */
async function initializeSources({
  dataViews,
  eventAnnotationService,
  datasourceMap,
  visualizationMap,
  visualizationState,
  datasourceStates,
  storage,
  defaultIndexPatternId,
  references,
  initialContext,
  adHocDataViews
}, options) {
  const annotationGroups = await initializeEventAnnotationGroups(eventAnnotationService, references);
  const {
    indexPatternRefs,
    indexPatterns
  } = await initializeDataViews({
    datasourceMap,
    datasourceStates,
    initialContext,
    dataViews,
    storage,
    defaultIndexPatternId,
    references,
    adHocDataViews,
    annotationGroups
  }, options);
  const initializedDatasourceStates = initializeDatasources({
    datasourceMap,
    datasourceStates,
    initialContext,
    indexPatternRefs,
    indexPatterns,
    references
  });
  return {
    indexPatterns,
    indexPatternRefs,
    annotationGroups,
    datasourceStates: initializedDatasourceStates,
    visualizationState: initializeVisualization({
      visualizationMap,
      visualizationState,
      datasourceStates,
      references,
      initialContext,
      annotationGroups
    })
  };
}
function initializeVisualization({
  visualizationMap,
  visualizationState,
  datasourceStates,
  references,
  annotationGroups
}) {
  if (visualizationState !== null && visualizationState !== void 0 && visualizationState.activeId) {
    var _visualizationMap$vis, _visualizationMap$vis2;
    return (_visualizationMap$vis = (_visualizationMap$vis2 = visualizationMap[visualizationState.activeId]) === null || _visualizationMap$vis2 === void 0 ? void 0 : _visualizationMap$vis2.initialize(() => '', visualizationState.state, COLORING_METHOD, datasourceStates,
    // initialize a new visualization with the color mapping off
    annotationGroups, references)) !== null && _visualizationMap$vis !== void 0 ? _visualizationMap$vis : visualizationState.state;
  }
  return visualizationState.state;
}
function initializeDatasources({
  datasourceMap,
  datasourceStates,
  indexPatternRefs,
  indexPatterns,
  references,
  initialContext
}) {
  // init datasources
  const states = {};
  for (const [datasourceId, datasource] of Object.entries(datasourceMap)) {
    if (datasourceStates[datasourceId]) {
      const state = datasource.initialize(datasourceStates[datasourceId].state || undefined, references, initialContext, indexPatternRefs, indexPatterns);
      states[datasourceId] = {
        isLoading: false,
        state
      };
    }
  }
  return states;
}
async function persistedStateToExpression(datasourceMap, visualizations, doc, services) {
  var _datasourceStates$dat2;
  const {
    state: {
      visualization: persistedVisualizationState,
      datasourceStates: persistedDatasourceStates,
      adHocDataViews,
      internalReferences
    },
    visualizationType,
    references,
    title,
    description
  } = doc;
  if (!visualizationType) {
    return {
      ast: null,
      indexPatterns: {},
      indexPatternRefs: [],
      activeVisualizationState: null,
      activeDatasourceState: null
    };
  }
  const annotationGroups = await initializeEventAnnotationGroups(services.eventAnnotationService, references);
  const visualization = visualizations[visualizationType];
  const datasourceStatesFromSO = Object.fromEntries(Object.entries(persistedDatasourceStates).map(([id, state]) => [id, {
    isLoading: false,
    state
  }]));
  const {
    indexPatterns,
    indexPatternRefs
  } = await initializeDataViews({
    datasourceMap,
    datasourceStates: datasourceStatesFromSO,
    references,
    dataViews: services.dataViews,
    storage: services.storage,
    defaultIndexPatternId: services.uiSettings.get('defaultIndex'),
    adHocDataViews,
    annotationGroups
  }, {
    isFullEditor: false
  });
  const datasourceStates = initializeDatasources({
    datasourceMap,
    datasourceStates: datasourceStatesFromSO,
    references: [...references, ...(internalReferences || [])],
    indexPatterns,
    indexPatternRefs
  });
  const activeVisualizationState = initializeVisualization({
    visualizationMap: visualizations,
    visualizationState: {
      state: persistedVisualizationState,
      activeId: visualizationType,
      selectedLayerId: null
    },
    datasourceStates,
    annotationGroups,
    references: [...references, ...(internalReferences || [])]
  });
  const datasourceLayers = (0, _utils2.getDatasourceLayers)(datasourceStates, datasourceMap, indexPatterns);
  const datasourceId = (0, _utils.getActiveDatasourceIdFromDoc)(doc);
  if (datasourceId == null) {
    return {
      ast: null,
      indexPatterns,
      indexPatternRefs,
      activeVisualizationState,
      activeDatasourceState: null
    };
  }
  const currentTimeRange = services.timefilter.getAbsoluteTime();
  return {
    ast: (0, _expression_helpers.buildExpression)({
      title,
      description,
      visualization,
      visualizationState: activeVisualizationState,
      datasourceMap,
      datasourceStates,
      datasourceLayers,
      indexPatterns,
      dateRange: {
        fromDate: currentTimeRange.from,
        toDate: currentTimeRange.to
      },
      forceDSL: services.forceDSL,
      nowInstant: services.nowProvider.get()
    }),
    activeVisualizationState,
    activeDatasourceState: (_datasourceStates$dat2 = datasourceStates[datasourceId]) === null || _datasourceStates$dat2 === void 0 ? void 0 : _datasourceStates$dat2.state,
    indexPatterns,
    indexPatternRefs
  };
}
function getMissingIndexPattern(currentDatasource, currentDatasourceState, indexPatterns) {
  if (currentDatasourceState !== null && currentDatasourceState !== void 0 && currentDatasourceState.isLoading || (currentDatasourceState === null || currentDatasourceState === void 0 ? void 0 : currentDatasourceState.state) == null || currentDatasource == null) {
    return [];
  }
  const missingIds = currentDatasource.checkIntegrity(currentDatasourceState.state, indexPatterns);
  if (!missingIds.length) {
    return [];
  }
  return missingIds;
}