"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.timeScaleOperation = void 0;
var _i18n = require("@kbn/i18n");
var _lensFormulaDocs = require("@kbn/lens-formula-docs");
var _utils = require("./utils");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const timeScaleOperation = exports.timeScaleOperation = {
  type: _lensFormulaDocs.NORMALIZE_BY_UNIT_ID,
  priority: 1,
  displayName: _lensFormulaDocs.NORMALIZE_BY_UNIT_NAME,
  input: 'fullReference',
  selectionStyle: 'hidden',
  requiredReferences: [{
    input: ['field', 'managedReference', 'fullReference'],
    validateMetadata: meta => meta.dataType === 'number' && !meta.isBucketed
  }],
  operationParams: [{
    name: 'unit',
    type: 'string',
    required: true
  }],
  getPossibleOperation: () => {
    return {
      dataType: 'number',
      isBucketed: false,
      scale: 'ratio'
    };
  },
  getDefaultLabel: (column, columns, indexPattern) => {
    return _lensFormulaDocs.NORMALIZE_BY_UNIT_ID;
  },
  toExpression: (layer, columnId) => {
    const currentColumn = layer.columns[columnId];
    const buckets = layer.columnOrder.filter(colId => layer.columns[colId].isBucketed);
    const dateColumn = buckets.find(colId => layer.columns[colId].operationType === 'date_histogram');
    return [{
      type: 'function',
      function: 'lens_time_scale',
      arguments: {
        dateColumnId: dateColumn ? [dateColumn] : [],
        inputColumnId: [currentColumn.references[0]],
        outputColumnId: [columnId],
        outputColumnName: [currentColumn.label],
        targetUnit: [currentColumn.params.unit],
        reducedTimeRange: currentColumn.reducedTimeRange ? [currentColumn.reducedTimeRange] : []
      }
    }];
  },
  buildColumn: ({
    referenceIds,
    previousColumn,
    layer,
    indexPattern
  }, columnParams) => {
    return {
      label: _lensFormulaDocs.NORMALIZE_BY_UNIT_NAME,
      dataType: 'number',
      operationType: _lensFormulaDocs.NORMALIZE_BY_UNIT_ID,
      isBucketed: false,
      references: referenceIds,
      params: {
        ...(0, _helpers.getFormatFromPreviousColumn)(previousColumn),
        unit: columnParams === null || columnParams === void 0 ? void 0 : columnParams.unit
      }
    };
  },
  isTransferable: () => {
    return true;
  },
  getErrorMessage: (layer, columnId) => {
    var _params$unit;
    const errors = (0, _utils.getErrorsForDateReference)(layer, columnId, _lensFormulaDocs.NORMALIZE_BY_UNIT_NAME);
    if (!layer.columns[columnId].params.unit) {
      errors.push({
        uniqueId: '',
        message: _i18n.i18n.translate('xpack.lens.indexPattern.timeScale.missingUnit', {
          defaultMessage: 'No unit specified for normalize by unit.'
        })
      });
    }
    if (['s', 'm', 'h', 'd'].indexOf((_params$unit = layer.columns[columnId].params.unit) !== null && _params$unit !== void 0 ? _params$unit : 's') === -1) {
      errors.push({
        uniqueId: '',
        message: _i18n.i18n.translate('xpack.lens.indexPattern.timeScale.wrongUnit', {
          defaultMessage: 'Unknown unit specified: use s, m, h or d.'
        })
      });
    }
    return errors;
  },
  filterable: false,
  shiftable: false
};