"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.emitTokenCountEstimateIfMissing = emitTokenCountEstimateIfMissing;
exports.manuallyCountTokens = manuallyCountTokens;
var _rxjs = require("rxjs");
var _inferenceCommon = require("@kbn/inference-common");
var _manually_count_tokens = require("./manually_count_tokens");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Operator mirroring the source and then emitting a tokenCount event when the source completes,
 * if and only if the source did not emit a tokenCount event itself.
 *
 * This is used to manually count tokens and emit the associated event for
 * providers that don't support sending token counts for the stream API.
 *
 * @param request the OpenAI request that was sent to the connector.
 */
function emitTokenCountEstimateIfMissing({
  request
}) {
  return source$ => {
    let tokenCountEmitted = false;
    const chunks = [];
    return new _rxjs.Observable(subscriber => {
      return source$.subscribe({
        next: value => {
          if ((0, _inferenceCommon.isChatCompletionTokenCountEvent)(value)) {
            tokenCountEmitted = true;
          } else if ((0, _inferenceCommon.isChatCompletionChunkEvent)(value)) {
            chunks.push(value);
          }
          subscriber.next(value);
        },
        error: err => {
          subscriber.error(err);
        },
        complete: () => {
          if (!tokenCountEmitted) {
            subscriber.next(manuallyCountTokens(request, chunks));
          }
          subscriber.complete();
        }
      });
    });
  };
}
function manuallyCountTokens(request, chunks) {
  const promptTokens = (0, _manually_count_tokens.manuallyCountPromptTokens)(request);
  const completionTokens = (0, _manually_count_tokens.manuallyCountCompletionTokens)(chunks);
  return {
    type: _inferenceCommon.ChatCompletionEventType.ChatCompletionTokenCount,
    tokens: {
      prompt: promptTokens,
      completion: completionTokens,
      total: promptTokens + completionTokens
    },
    ...(request.model ? {
      model: request.model
    } : {})
  };
}