"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._runBulkUpgradeTask = _runBulkUpgradeTask;
exports.scheduleBulkUpgrade = scheduleBulkUpgrade;
var _common = require("@kbn/spaces-plugin/common");
var _http_authorization_header = require("../../../common/http_authorization_header");
var _packages = require("../../services/epm/packages");
var _services = require("../../services");
var _constants = require("../../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function _runBulkUpgradeTask({
  abortController,
  taskParams,
  logger
}) {
  const {
    packages,
    spaceId = _common.DEFAULT_SPACE_ID,
    authorizationHeader,
    force,
    prerelease,
    upgradePackagePolicies
  } = taskParams;
  const esClient = _services.appContextService.getInternalUserESClient();
  const savedObjectsClient = _services.appContextService.getInternalUserSOClientForSpaceId(spaceId);
  const results = [];
  for (const pkg of packages) {
    // Throw between package install if task is aborted
    if (abortController.signal.aborted) {
      throw new Error('Task was aborted');
    }
    try {
      const installResult = await (0, _packages.installPackage)({
        spaceId,
        authorizationHeader: authorizationHeader ? new _http_authorization_header.HTTPAuthorizationHeader(authorizationHeader.scheme, authorizationHeader.credentials, authorizationHeader.username) : undefined,
        installSource: 'registry',
        // Upgrade can only happens from the registry,
        esClient,
        savedObjectsClient,
        pkgkey: pkg !== null && pkg !== void 0 && pkg.version ? `${pkg.name}-${pkg.version}` : pkg.name,
        force,
        prerelease
      });
      if (installResult.error) {
        throw installResult.error;
      }
      if (upgradePackagePolicies) {
        await bulkUpgradePackagePolicies({
          savedObjectsClient,
          esClient,
          pkgName: pkg.name
        });
      }
      results.push({
        name: pkg.name,
        success: true
      });
    } catch (error) {
      logger.error(`Upgrade of package: ${pkg.name} failed`, {
        error
      });
      results.push({
        name: pkg.name,
        success: false,
        error: (0, _utils.formatError)(error)
      });
    }
  }
  return results;
}
async function bulkUpgradePackagePolicies({
  savedObjectsClient,
  esClient,
  pkgName
}) {
  const policyIdsToUpgrade = await _services.packagePolicyService.listIds(savedObjectsClient, {
    page: 1,
    perPage: _constants.SO_SEARCH_LIMIT,
    kuery: `${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${pkgName}`
  });
  if (policyIdsToUpgrade.items.length) {
    const upgradePackagePoliciesResults = await _services.packagePolicyService.bulkUpgrade(savedObjectsClient, esClient, policyIdsToUpgrade.items);
    const errors = upgradePackagePoliciesResults.filter(result => !result.success).map(result => {
      var _result$body$message, _result$body;
      return `${result.statusCode}: ${(_result$body$message = (_result$body = result.body) === null || _result$body === void 0 ? void 0 : _result$body.message) !== null && _result$body$message !== void 0 ? _result$body$message : ''}`;
    });
    if (errors.length) {
      throw new Error(`Package policies upgrade for ${pkgName} failed:\n${errors.join('\n')}`);
    }
  }
}
async function scheduleBulkUpgrade(taskManagerStart, taskParams) {
  return (0, _utils.scheduleBulkOperationTask)(taskManagerStart, {
    ...taskParams,
    type: 'bulk_upgrade'
  });
}