"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cleanPreconfiguredFleetServerHosts = cleanPreconfiguredFleetServerHosts;
exports.createCloudFleetServerHostIfNeeded = createCloudFleetServerHostIfNeeded;
exports.createOrUpdatePreconfiguredFleetServerHosts = createOrUpdatePreconfiguredFleetServerHosts;
exports.ensurePreconfiguredFleetServerHosts = ensurePreconfiguredFleetServerHosts;
exports.getCloudFleetServersHosts = getCloudFleetServersHosts;
exports.getPreconfiguredFleetServerHostFromConfig = getPreconfiguredFleetServerHostFromConfig;
var _services = require("../../../common/services");
var _constants = require("../../constants");
var _errors = require("../../errors");
var _app_context = require("../app_context");
var _fleet_server_host = require("../fleet_server_host");
var _agentless = require("../utils/agentless");
var _agent_policy = require("../agent_policy");
var _utils = require("./utils");
var _outputs = require("./outputs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getCloudFleetServersHosts() {
  const cloudSetup = _app_context.appContextService.getCloud();
  if (cloudSetup && !cloudSetup.isServerlessEnabled && cloudSetup.isCloudEnabled && cloudSetup.cloudHost) {
    // Fleet Server url are formed like this `https://<deploymentId>.fleet.<host>
    return [`https://${cloudSetup.deploymentId}.fleet.${cloudSetup.cloudHost}${cloudSetup.cloudDefaultPort && cloudSetup.cloudDefaultPort !== '443' ? `:${cloudSetup.cloudDefaultPort}` : ''}`];
  }
}
function getPreconfiguredFleetServerHostFromConfig(config) {
  const {
    fleetServerHosts: fleetServerHostsFromConfig
  } = config;
  const legacyFleetServerHostsConfig = getConfigFleetServerHosts(config);
  const cloudServerHosts = getCloudFleetServersHosts();
  const fleetServerHosts = (fleetServerHostsFromConfig || []).concat([...(legacyFleetServerHostsConfig ? [{
    name: 'Default',
    is_default: true,
    id: _constants.DEFAULT_FLEET_SERVER_HOST_ID,
    host_urls: legacyFleetServerHostsConfig
  }] : []),
  // Include agentless Fleet Server host in ECH
  ...((0, _agentless.isAgentlessEnabled)() && cloudServerHosts ? [{
    id: _constants.ECH_AGENTLESS_FLEET_SERVER_HOST_ID,
    name: 'Internal Fleet Server for agentless',
    host_urls: cloudServerHosts,
    is_default: false,
    is_preconfigured: true
  }] : [])]);
  if (fleetServerHosts.filter(fleetServerHost => fleetServerHost.is_default).length > 1) {
    throw new _errors.FleetError('Only one default Fleet Server host is allowed');
  }
  return fleetServerHosts;
}
async function ensurePreconfiguredFleetServerHosts(soClient, esClient, preconfiguredFleetServerHosts) {
  await createOrUpdatePreconfiguredFleetServerHosts(soClient, esClient, preconfiguredFleetServerHosts);
  await createCloudFleetServerHostIfNeeded(soClient, esClient);
  await cleanPreconfiguredFleetServerHosts(soClient, esClient, preconfiguredFleetServerHosts);
}
async function createOrUpdatePreconfiguredFleetServerHosts(soClient, esClient, preconfiguredFleetServerHosts) {
  const existingFleetServerHosts = await _fleet_server_host.fleetServerHostService.bulkGet(preconfiguredFleetServerHosts.map(({
    id
  }) => id), {
    ignoreNotFound: true
  });
  await Promise.all(preconfiguredFleetServerHosts.map(async preconfiguredFleetServerHost => {
    const existingHost = existingFleetServerHosts.find(fleetServerHost => fleetServerHost.id === preconfiguredFleetServerHost.id);
    const {
      id,
      ...data
    } = preconfiguredFleetServerHost;
    const isCreate = !existingHost;
    const isUpdateWithNewData = existingHost && (!existingHost.is_preconfigured || (await isPreconfiguredFleetServerHostDifferentFromCurrent(existingHost, preconfiguredFleetServerHost)));
    const secretHashes = await hashSecrets(preconfiguredFleetServerHost);
    if (isCreate) {
      await _fleet_server_host.fleetServerHostService.create(soClient, esClient, {
        ...data,
        is_preconfigured: true
      }, {
        id,
        overwrite: true,
        fromPreconfiguration: true,
        secretHashes
      });
    } else if (isUpdateWithNewData) {
      await _fleet_server_host.fleetServerHostService.update(soClient, esClient, id, {
        ...data,
        is_preconfigured: true
      }, {
        fromPreconfiguration: true,
        secretHashes
      });
      if (data.is_default) {
        await _agent_policy.agentPolicyService.bumpAllAgentPolicies(esClient);
      } else {
        await _agent_policy.agentPolicyService.bumpAllAgentPoliciesForFleetServerHosts(esClient, id);
      }
    }
  }));
}
async function createCloudFleetServerHostIfNeeded(soClient, esClient) {
  const cloudServerHosts = getCloudFleetServersHosts();
  if (!cloudServerHosts || cloudServerHosts.length === 0) {
    return;
  }
  const defaultFleetServerHost = await _fleet_server_host.fleetServerHostService.getDefaultFleetServerHost();
  if (!defaultFleetServerHost) {
    await _fleet_server_host.fleetServerHostService.create(soClient, esClient, {
      name: 'Default',
      is_default: true,
      host_urls: cloudServerHosts,
      is_preconfigured: false
    }, {
      id: _constants.DEFAULT_FLEET_SERVER_HOST_ID,
      overwrite: true,
      fromPreconfiguration: true
    });
  }
}
async function cleanPreconfiguredFleetServerHosts(soClient, esClient, preconfiguredFleetServerHosts) {
  const existingFleetServerHosts = await _fleet_server_host.fleetServerHostService.list();
  const existingPreconfiguredHosts = existingFleetServerHosts.items.filter(o => o.is_preconfigured === true);
  for (const existingFleetServerHost of existingPreconfiguredHosts) {
    const hasBeenDelete = !preconfiguredFleetServerHosts.find(({
      id
    }) => existingFleetServerHost.id === id);
    if (!hasBeenDelete) {
      continue;
    }
    if (existingFleetServerHost.is_default) {
      await _fleet_server_host.fleetServerHostService.update(soClient, esClient, existingFleetServerHost.id, {
        is_preconfigured: false
      }, {
        fromPreconfiguration: true
      });
    } else {
      await _fleet_server_host.fleetServerHostService.delete(esClient, existingFleetServerHost.id, {
        fromPreconfiguration: true
      });
    }
  }
}
function getConfigFleetServerHosts(config) {
  var _config$agents, _config$agents$fleet_, _config$agents2, _config$agents2$fleet;
  return config !== null && config !== void 0 && (_config$agents = config.agents) !== null && _config$agents !== void 0 && (_config$agents$fleet_ = _config$agents.fleet_server) !== null && _config$agents$fleet_ !== void 0 && _config$agents$fleet_.hosts && config.agents.fleet_server.hosts.length > 0 ? config === null || config === void 0 ? void 0 : (_config$agents2 = config.agents) === null || _config$agents2 === void 0 ? void 0 : (_config$agents2$fleet = _config$agents2.fleet_server) === null || _config$agents2$fleet === void 0 ? void 0 : _config$agents2$fleet.hosts : undefined;
}
async function hashSecrets(preconfiguredFleetServerHost) {
  var _preconfiguredFleetSe, _preconfiguredFleetSe2, _preconfiguredFleetSe5, _preconfiguredFleetSe6, _preconfiguredFleetSe9, _preconfiguredFleetSe10;
  let secrets = {};
  if (typeof ((_preconfiguredFleetSe = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe === void 0 ? void 0 : (_preconfiguredFleetSe2 = _preconfiguredFleetSe.ssl) === null || _preconfiguredFleetSe2 === void 0 ? void 0 : _preconfiguredFleetSe2.key) === 'string') {
    var _preconfiguredFleetSe3, _preconfiguredFleetSe4;
    const key = await (0, _outputs.hashSecret)((_preconfiguredFleetSe3 = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe3 === void 0 ? void 0 : (_preconfiguredFleetSe4 = _preconfiguredFleetSe3.ssl) === null || _preconfiguredFleetSe4 === void 0 ? void 0 : _preconfiguredFleetSe4.key);
    secrets = {
      ssl: {
        key
      }
    };
  }
  if (typeof ((_preconfiguredFleetSe5 = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe5 === void 0 ? void 0 : (_preconfiguredFleetSe6 = _preconfiguredFleetSe5.ssl) === null || _preconfiguredFleetSe6 === void 0 ? void 0 : _preconfiguredFleetSe6.key) === 'string') {
    var _preconfiguredFleetSe7, _preconfiguredFleetSe8;
    const esKey = await (0, _outputs.hashSecret)((_preconfiguredFleetSe7 = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe7 === void 0 ? void 0 : (_preconfiguredFleetSe8 = _preconfiguredFleetSe7.ssl) === null || _preconfiguredFleetSe8 === void 0 ? void 0 : _preconfiguredFleetSe8.key);
    secrets = {
      ...(secrets ? secrets : {}),
      ssl: {
        es_key: esKey
      }
    };
  }
  if (typeof ((_preconfiguredFleetSe9 = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe9 === void 0 ? void 0 : (_preconfiguredFleetSe10 = _preconfiguredFleetSe9.ssl) === null || _preconfiguredFleetSe10 === void 0 ? void 0 : _preconfiguredFleetSe10.agent_key) === 'string') {
    var _preconfiguredFleetSe11, _preconfiguredFleetSe12;
    const agentKey = await (0, _outputs.hashSecret)((_preconfiguredFleetSe11 = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe11 === void 0 ? void 0 : (_preconfiguredFleetSe12 = _preconfiguredFleetSe11.ssl) === null || _preconfiguredFleetSe12 === void 0 ? void 0 : _preconfiguredFleetSe12.agent_key);
    secrets = {
      ...(secrets ? secrets : {}),
      ssl: {
        agent_key: agentKey
      }
    };
  }
  return secrets;
}
async function isPreconfiguredFleetServerHostDifferentFromCurrent(existingFleetServerHost, preconfiguredFleetServerHost) {
  var _preconfiguredFleetSe19;
  const secretFieldsAreDifferent = async () => {
    var _preconfiguredFleetSe13, _preconfiguredFleetSe14, _existingFleetServerH, _existingFleetServerH2, _preconfiguredFleetSe15, _preconfiguredFleetSe16, _existingFleetServerH3, _existingFleetServerH4, _preconfiguredFleetSe17, _preconfiguredFleetSe18, _existingFleetServerH5, _existingFleetServerH6;
    const sslKeyHashIsDifferent = await (0, _outputs.isSecretDifferent)((_preconfiguredFleetSe13 = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe13 === void 0 ? void 0 : (_preconfiguredFleetSe14 = _preconfiguredFleetSe13.ssl) === null || _preconfiguredFleetSe14 === void 0 ? void 0 : _preconfiguredFleetSe14.key, (_existingFleetServerH = existingFleetServerHost.secrets) === null || _existingFleetServerH === void 0 ? void 0 : (_existingFleetServerH2 = _existingFleetServerH.ssl) === null || _existingFleetServerH2 === void 0 ? void 0 : _existingFleetServerH2.key);
    const sslESKeyHashIsDifferent = await (0, _outputs.isSecretDifferent)((_preconfiguredFleetSe15 = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe15 === void 0 ? void 0 : (_preconfiguredFleetSe16 = _preconfiguredFleetSe15.ssl) === null || _preconfiguredFleetSe16 === void 0 ? void 0 : _preconfiguredFleetSe16.es_key, (_existingFleetServerH3 = existingFleetServerHost.secrets) === null || _existingFleetServerH3 === void 0 ? void 0 : (_existingFleetServerH4 = _existingFleetServerH3.ssl) === null || _existingFleetServerH4 === void 0 ? void 0 : _existingFleetServerH4.es_key);
    const sslAgentKeyHashIsDifferent = await (0, _outputs.isSecretDifferent)((_preconfiguredFleetSe17 = preconfiguredFleetServerHost.secrets) === null || _preconfiguredFleetSe17 === void 0 ? void 0 : (_preconfiguredFleetSe18 = _preconfiguredFleetSe17.ssl) === null || _preconfiguredFleetSe18 === void 0 ? void 0 : _preconfiguredFleetSe18.agent_key, (_existingFleetServerH5 = existingFleetServerHost.secrets) === null || _existingFleetServerH5 === void 0 ? void 0 : (_existingFleetServerH6 = _existingFleetServerH5.ssl) === null || _existingFleetServerH6 === void 0 ? void 0 : _existingFleetServerH6.agent_key);
    return sslKeyHashIsDifferent || sslESKeyHashIsDifferent || sslAgentKeyHashIsDifferent;
  };
  return existingFleetServerHost.is_default !== preconfiguredFleetServerHost.is_default || existingFleetServerHost.name !== preconfiguredFleetServerHost.name || (0, _utils.isDifferent)(existingFleetServerHost.is_internal, preconfiguredFleetServerHost.is_internal) || (0, _utils.isDifferent)(existingFleetServerHost.host_urls.map(_services.normalizeHostsForAgents), preconfiguredFleetServerHost === null || preconfiguredFleetServerHost === void 0 ? void 0 : (_preconfiguredFleetSe19 = preconfiguredFleetServerHost.host_urls) === null || _preconfiguredFleetSe19 === void 0 ? void 0 : _preconfiguredFleetSe19.map(_services.normalizeHostsForAgents)) || (0, _utils.isDifferent)(existingFleetServerHost === null || existingFleetServerHost === void 0 ? void 0 : existingFleetServerHost.proxy_id, preconfiguredFleetServerHost.proxy_id) || (0, _utils.isDifferent)(existingFleetServerHost === null || existingFleetServerHost === void 0 ? void 0 : existingFleetServerHost.ssl, preconfiguredFleetServerHost === null || preconfiguredFleetServerHost === void 0 ? void 0 : preconfiguredFleetServerHost.ssl) || secretFieldsAreDifferent();
}