"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.downloadSourceService = void 0;
var _lodash = require("lodash");
var _common = require("@kbn/kibana-utils-plugin/common");
var _constants = require("../constants");
var _errors = require("../errors");
var _common2 = require("../../common");
var _secrets = require("./secrets");
var _agent_policy = require("./agent_policy");
var _app_context = require("./app_context");
var _saved_object = require("./saved_object");
var _fleet_proxies = require("./fleet_proxies");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function savedObjectToDownloadSource(so) {
  const {
    ssl,
    source_id: sourceId,
    ...attributes
  } = so.attributes;
  return {
    id: sourceId !== null && sourceId !== void 0 ? sourceId : so.id,
    ...attributes,
    ...(ssl ? {
      ssl: JSON.parse(ssl)
    } : {})
  };
}
const fakeRequest = {
  headers: {},
  getBasePath: () => '',
  path: '/',
  route: {
    settings: {}
  },
  url: {
    href: '/'
  },
  raw: {
    req: {
      url: '/'
    }
  }
};
class DownloadSourceService {
  get soClient() {
    return _app_context.appContextService.getInternalUserSOClient(fakeRequest);
  }
  get encryptedSoClient() {
    return _app_context.appContextService.getEncryptedSavedObjects();
  }
  async get(id) {
    const soResponse = await this.encryptedSoClient.getDecryptedAsInternalUser(_constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE, id);
    if (soResponse.error) {
      throw new _errors.FleetError(soResponse.error.message);
    }
    return savedObjectToDownloadSource(soResponse);
  }
  async list() {
    const downloadSourcesFinder = await this.encryptedSoClient.createPointInTimeFinderDecryptedAsInternalUser({
      type: _constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE,
      perPage: _common2.SO_SEARCH_LIMIT,
      sortField: 'is_default',
      sortOrder: 'desc'
    });
    let downloadSources = [];
    let total = 0;
    let page = 0;
    let perPage = 0;
    for await (const result of downloadSourcesFinder.find()) {
      downloadSources = result.saved_objects;
      total = result.total;
      page = result.page;
      perPage = result.per_page;
      break; // Return first page;
    }
    await downloadSourcesFinder.close();
    return {
      items: downloadSources.map(savedObjectToDownloadSource),
      total,
      page,
      perPage
    };
  }
  async create(soClient, esClient, downloadSource, options) {
    var _appContextService$ge, _options$overwrite;
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`Creating new download source`);
    const data = {
      ...(0, _lodash.omit)(downloadSource, ['ssl', 'secrets'])
    };
    if (!((_appContextService$ge = _app_context.appContextService.getEncryptedSavedObjectsSetup()) !== null && _appContextService$ge !== void 0 && _appContextService$ge.canEncrypt)) {
      throw new _errors.FleetEncryptedSavedObjectEncryptionKeyRequired(`Agent binary source needs encrypted saved object api key to be set`);
    }
    await this.requireUniqueName({
      name: downloadSource.name,
      id: options === null || options === void 0 ? void 0 : options.id
    });
    if (data.proxy_id) {
      await this.throwIfProxyNotFound(soClient, data.proxy_id);
    }

    // default should be only one
    if (data.is_default) {
      const defaultDownloadSourceId = await this.getDefaultDownloadSourceId();
      if (defaultDownloadSourceId) {
        await this.update(soClient, esClient, defaultDownloadSourceId, {
          is_default: false
        });
      }
    }
    if (options !== null && options !== void 0 && options.id) {
      data.source_id = options === null || options === void 0 ? void 0 : options.id;
    }
    if (downloadSource.ssl) {
      data.ssl = JSON.stringify(downloadSource.ssl);
    }
    // Store secret values if enabled; if not, store plain text values
    if (await (0, _secrets.isSSLSecretStorageEnabled)(esClient, soClient)) {
      const {
        downloadSource: downloadSourceWithSecrets
      } = await (0, _secrets.extractAndWriteDownloadSourcesSecrets)({
        downloadSource,
        esClient
      });
      if (downloadSourceWithSecrets.secrets) data.secrets = downloadSourceWithSecrets.secrets;
    } else {
      var _downloadSource$ssl, _downloadSource$secre, _downloadSource$secre2;
      if (!((_downloadSource$ssl = downloadSource.ssl) !== null && _downloadSource$ssl !== void 0 && _downloadSource$ssl.key) && (_downloadSource$secre = downloadSource.secrets) !== null && _downloadSource$secre !== void 0 && (_downloadSource$secre2 = _downloadSource$secre.ssl) !== null && _downloadSource$secre2 !== void 0 && _downloadSource$secre2.key) {
        data.ssl = JSON.stringify({
          ...downloadSource.ssl,
          ...downloadSource.secrets.ssl
        });
      }
    }
    const newSo = await this.soClient.create(_constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE, data, {
      id: options === null || options === void 0 ? void 0 : options.id,
      overwrite: (_options$overwrite = options === null || options === void 0 ? void 0 : options.overwrite) !== null && _options$overwrite !== void 0 ? _options$overwrite : false
    });
    logger.debug(`Creating new download source ${options === null || options === void 0 ? void 0 : options.id}`);
    // soClient.create doesn't return the decrypted attributes, so we need to fetch it again.
    const retrievedSo = await this.encryptedSoClient.getDecryptedAsInternalUser(_constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE, newSo.id);
    return savedObjectToDownloadSource(retrievedSo);
  }
  async update(soClient, esClient, id, newData) {
    let secretsToDelete = [];
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`Updating download source ${id} with ${newData}`);
    const originalItem = await this.get(id);
    const updateData = {
      ...(0, _lodash.omit)(newData, ['ssl', 'secrets'])
    };
    if (updateData.proxy_id) {
      await this.throwIfProxyNotFound(soClient, updateData.proxy_id);
    }
    if (updateData.name) {
      await this.requireUniqueName({
        name: updateData.name,
        id
      });
    }
    if (newData.ssl) {
      updateData.ssl = JSON.stringify(newData.ssl);
    } else if (newData.ssl === null) {
      // Explicitly set to null to allow to delete the field
      updateData.ssl = null;
    }
    if (updateData.is_default) {
      const defaultDownloadSourceId = await this.getDefaultDownloadSourceId();
      if (defaultDownloadSourceId && defaultDownloadSourceId !== id) {
        await this.update(soClient, esClient, defaultDownloadSourceId, {
          is_default: false
        });
      }
    }
    // Store secret values if enabled; if not, store plain text values
    if (await (0, _secrets.isSecretStorageEnabled)(esClient, soClient)) {
      const secretsRes = await (0, _secrets.extractAndUpdateDownloadSourceSecrets)({
        oldDownloadSource: originalItem,
        downloadSourceUpdate: newData,
        esClient
      });
      updateData.secrets = secretsRes.downloadSourceUpdate.secrets;
      secretsToDelete = secretsRes.secretsToDelete;
    } else {
      var _newData$ssl, _newData$secrets, _newData$secrets$ssl;
      if (!((_newData$ssl = newData.ssl) !== null && _newData$ssl !== void 0 && _newData$ssl.key) && (_newData$secrets = newData.secrets) !== null && _newData$secrets !== void 0 && (_newData$secrets$ssl = _newData$secrets.ssl) !== null && _newData$secrets$ssl !== void 0 && _newData$secrets$ssl.key) {
        updateData.ssl = JSON.stringify({
          ...newData.ssl,
          ...newData.secrets.ssl
        });
      }
    }
    if (secretsToDelete.length) {
      try {
        await (0, _secrets.deleteSecrets)({
          esClient,
          ids: secretsToDelete.map(s => s.id)
        });
      } catch (err) {
        logger.warn(`Error cleaning up secrets for output ${id}: ${err.message}`);
      }
    }
    const soResponse = await this.soClient.update(_constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE, id, updateData);
    if (soResponse.error) {
      throw new _errors.FleetError(soResponse.error.message);
    } else {
      logger.debug(`Updated download source ${id}`);
    }
  }
  async delete(id) {
    const logger = _app_context.appContextService.getLogger();
    logger.debug(`Deleting download source ${id}`);
    const targetDS = await this.get(id);
    if (targetDS.is_default) {
      throw new _errors.DownloadSourceError(`Default Download source ${id} cannot be deleted.`);
    }
    await _agent_policy.agentPolicyService.removeDefaultSourceFromAll(_app_context.appContextService.getInternalUserESClient(), id);
    await (0, _secrets.deleteDownloadSourceSecrets)({
      esClient: _app_context.appContextService.getInternalUserESClient(),
      downloadSource: targetDS
    });
    logger.debug(`Deleting download source ${id}`);
    return this.soClient.delete(_constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE, id);
  }
  async getDefaultDownloadSourceId() {
    const results = await this._getDefaultDownloadSourceSO();
    if (!results.saved_objects.length) {
      return null;
    }
    return savedObjectToDownloadSource(results.saved_objects[0]).id;
  }
  async ensureDefault(soClient, esClient) {
    const downloadSources = await this.list();
    const defaultDS = downloadSources.items.find(o => o.is_default);
    if (!defaultDS) {
      const newDefaultDS = {
        name: 'Elastic Artifacts',
        is_default: true,
        host: _constants.DEFAULT_DOWNLOAD_SOURCE_URI
      };
      return await this.create(soClient, esClient, newDefaultDS, {
        id: _constants.DEFAULT_DOWNLOAD_SOURCE_ID,
        overwrite: true
      });
    }
    return defaultDS;
  }
  async requireUniqueName(downloadSource) {
    const results = await this.soClient.find({
      type: _constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE,
      searchFields: ['name'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(downloadSource.name)
    });
    const idsWithName = results.total && results.saved_objects.map(({
      id
    }) => id);
    if (Array.isArray(idsWithName)) {
      const isEditingSelf = (downloadSource === null || downloadSource === void 0 ? void 0 : downloadSource.id) && idsWithName.includes(downloadSource.id);
      if (!downloadSource.id || !isEditingSelf) {
        const isSingle = idsWithName.length === 1;
        const existClause = isSingle ? `Download Source '${idsWithName[0]}' already exists` : `Download Sources '${idsWithName.join(',')}' already exist`;
        throw new _errors.FleetError(`${existClause} with name '${downloadSource.name}'`);
      }
    }
  }
  async listAllForProxyId(proxyId) {
    const downloadSources = await this.soClient.find({
      type: _constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE,
      searchFields: ['proxy_id'],
      search: proxyId,
      perPage: _common2.SO_SEARCH_LIMIT
    });
    return {
      items: downloadSources.saved_objects.map(savedObjectToDownloadSource),
      total: downloadSources.total
    };
  }
  async throwIfProxyNotFound(soClient, id) {
    try {
      await (0, _fleet_proxies.getFleetProxy)(soClient, id);
    } catch (err) {
      if (err instanceof _common.SavedObjectNotFound) {
        throw new _errors.DownloadSourceError(`Proxy ${id} not found`);
      }
      throw new _errors.DownloadSourceError(`Error checking proxy_id: ${err.message}`);
    }
  }
  async _getDefaultDownloadSourceSO() {
    return await this.soClient.find({
      type: _constants.DOWNLOAD_SOURCE_SAVED_OBJECT_TYPE,
      searchFields: ['is_default'],
      search: 'true'
    });
  }
}
const downloadSourceService = exports.downloadSourceService = new DownloadSourceService();