"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getValueFromAccessor = exports.getMinValue = exports.getMaxValue = exports.getGoalValue = exports.getAccessorsFromArgs = void 0;
var _chartExpressionsCommon = require("@kbn/chart-expressions-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getValueFromAccessor = (accessor, row) => {
  if (!row || !accessor) return;
  const value = accessor && row[accessor];
  if (value === null || Array.isArray(value) && !value.length) {
    return;
  }
  if (typeof value === 'number') {
    return value;
  }
  if (Array.isArray(value) && typeof value[value.length - 1] === 'number') {
    return value[value.length - 1];
  }
};

// returns nice rounded numbers similar to d3 nice() function
exports.getValueFromAccessor = getValueFromAccessor;
function getNiceRange(min, max) {
  const maxTicks = 5;
  const offsetMax = max + 0.0000001; // added to avoid max value equal to metric value
  const range = getNiceNumber(offsetMax - min);
  const tickSpacing = getNiceNumber(range / (maxTicks - 1));
  return {
    min: Math.floor(min / tickSpacing) * tickSpacing,
    max: Math.ceil(Math.ceil(offsetMax / tickSpacing) * tickSpacing)
  };
}
function getNiceNumber(localRange) {
  const exponent = Math.floor(Math.log10(localRange));
  const fraction = localRange / Math.pow(10, exponent);
  let niceFraction = 10;
  if (fraction <= 1) niceFraction = 1;else if (fraction <= 2) niceFraction = 2;else if (fraction <= 5) niceFraction = 5;
  return niceFraction * Math.pow(10, exponent);
}
const getMaxValue = (row, accessors, paletteParams, isRespectRanges) => {
  const FALLBACK_VALUE = 100;
  const currentValue = accessors !== null && accessors !== void 0 && accessors.max ? getValueFromAccessor(accessors.max, row) : undefined;
  if (currentValue !== undefined && currentValue !== null) {
    return currentValue;
  }
  if (isRespectRanges && paletteParams !== null && paletteParams !== void 0 && paletteParams.rangeMax) {
    const metricValue = accessors !== null && accessors !== void 0 && accessors.metric ? getValueFromAccessor(accessors.metric, row) : undefined;
    return !metricValue || metricValue < (paletteParams === null || paletteParams === void 0 ? void 0 : paletteParams.rangeMax) ? paletteParams.rangeMax : metricValue;
  }
  if (row && accessors) {
    const {
      metric,
      goal
    } = accessors;
    const metricValue = metric && row[metric];
    const goalValue = goal && row[goal];
    const minValue = getMinValue(row, accessors, paletteParams, isRespectRanges);
    if (metricValue != null) {
      const numberValues = [minValue, goalValue, metricValue].filter(v => typeof v === 'number');
      const maxValue = Math.max(...numberValues);
      return getNiceRange(minValue, maxValue).max;
    }
  }
  return FALLBACK_VALUE;
};
exports.getMaxValue = getMaxValue;
const getMinValue = (row, accessors, paletteParams, isRespectRanges) => {
  const currentValue = accessors !== null && accessors !== void 0 && accessors.min ? getValueFromAccessor(accessors.min, row) : undefined;
  if (currentValue != null) {
    return currentValue;
  }
  if (isRespectRanges && paletteParams !== null && paletteParams !== void 0 && paletteParams.rangeMin) {
    const metricValue = accessors !== null && accessors !== void 0 && accessors.metric ? getValueFromAccessor(accessors.metric, row) : undefined;
    return !metricValue || metricValue > (paletteParams === null || paletteParams === void 0 ? void 0 : paletteParams.rangeMin) ? paletteParams.rangeMin : metricValue;
  }
  const FALLBACK_VALUE = 0;
  if (row && accessors) {
    const {
      metric,
      max
    } = accessors;
    const metricValue = metric && row[metric];
    const maxValue = max && row[max];
    const numberValues = [metricValue, maxValue].filter(v => typeof v === 'number');
    if (Math.min(...numberValues) <= 0) {
      return Math.min(...numberValues) - 10; // TODO: TO THINK THROUGH
    }
  }
  return FALLBACK_VALUE;
};
exports.getMinValue = getMinValue;
const getGoalValue = (row, accessors) => {
  const currentValue = accessors !== null && accessors !== void 0 && accessors.goal ? getValueFromAccessor(accessors.goal, row) : undefined;
  if (currentValue != null) {
    return currentValue;
  }
  const minValue = getMinValue(row, accessors);
  const maxValue = getMaxValue(row, accessors);
  return Math.round((maxValue - minValue) * 0.75 + minValue);
};
exports.getGoalValue = getGoalValue;
const getAccessorsFromArgs = (args, columns) => {
  const {
    metric,
    min,
    max,
    goal
  } = args;
  if (!metric && !min && !max && !goal) {
    return;
  }
  return {
    min: min ? (0, _chartExpressionsCommon.getAccessorByDimension)(min, columns) : undefined,
    max: max ? (0, _chartExpressionsCommon.getAccessorByDimension)(max, columns) : undefined,
    goal: goal ? (0, _chartExpressionsCommon.getAccessorByDimension)(goal, columns) : undefined,
    metric: metric ? (0, _chartExpressionsCommon.getAccessorByDimension)(metric, columns) : undefined
  };
};
exports.getAccessorsFromArgs = getAccessorsFromArgs;