"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PARAM_TYPES_NO_NEED_IMPLICIT_STRING_CASTING = void 0;
exports.appendToESQLQuery = appendToESQLQuery;
exports.escapeStringValue = escapeStringValue;
exports.extractMatchFunctionDetails = extractMatchFunctionDetails;
exports.getOperator = void 0;
exports.getSupportedOperators = getSupportedOperators;
var _esqlAst = require("@kbn/esql-ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const PARAM_TYPES_NO_NEED_IMPLICIT_STRING_CASTING = exports.PARAM_TYPES_NO_NEED_IMPLICIT_STRING_CASTING = ['date', 'date_nanos', 'version', 'ip', 'boolean', 'number', 'string'];

/**
 * Gets the operator and expression type for the given operation
 */
const getOperator = operation => {
  switch (operation) {
    case 'is_not_null':
      return {
        operator: 'is not null',
        expressionType: 'postfix-unary'
      };
    case 'is_null':
      return {
        operator: 'is null',
        expressionType: 'postfix-unary'
      };
    case '-':
      return {
        operator: '!=',
        expressionType: 'binary'
      };
    default:
      return {
        operator: '==',
        expressionType: 'binary'
      };
  }
};

/**
 * Get the list of supported operators dynamically by mapping all possible operation inputs
 */
exports.getOperator = getOperator;
function getSupportedOperators() {
  const operations = ['+', '-', 'is_not_null', 'is_null'];
  return operations.map(op => getOperator(op).operator);
}

/**
 * Escapes a string value for use in ES|QL queries by escaping backslashes and quotes
 */
function escapeStringValue(val) {
  return `"${val.replace(/\\/g, '\\\\').replace(/\"/g, '\\"')}"`;
}

/**
 * Append in a new line the appended text to take care of the case where the user adds a comment at the end of the query.
 * In these cases a base query such as "from index // comment" will result in errors or wrong data if we don't append in a new line
 */
function appendToESQLQuery(baseESQLQuery, appendedText) {
  return `${baseESQLQuery}\n${appendedText}`;
}

/**
 * Extracts field name and value from a MATCH function AST node
 */
function extractMatchFunctionDetails(matchFunction) {
  const args = matchFunction.args;
  const [column] = args.filter(a => (0, _esqlAst.isColumn)(a));
  const [valueNode] = args.filter(a => (0, _esqlAst.isStringLiteral)(a));
  if (column && valueNode) {
    const columnName = column.name;
    const literalValue = valueNode.valueUnquoted;
    return {
      columnName,
      literalValue
    };
  }
  return null;
}