"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.str = exports.srcAs = exports.src = exports.par = exports.num = exports.kwd = exports.int = exports.float = exports.dpar = exports.col = exports.bool = void 0;
var _builder = require("../../ast/builder");
var _holes = require("./holes");
var _synth_literal_fragment = require("./synth_literal_fragment");
var _synth_node = require("./synth_node");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Creates an ES|QL source node.
 *
 * @param index Elasticsearch index name to create a source node for.
 * @returns ES|QL source node.
 */
const src = (index, prefix, selector) => {
  const node = _builder.Builder.expression.source.index(index, prefix, selector);
  return _synth_node.SynthNode.from(node);
};

/**
 * Creates an ES|QL source node with an alias (e.g., `index AS alias`).
 *
 * @param index Elasticsearch index name to create a source node for.
 * @param alias The alias name for the source.
 * @returns ES|QL binary expression node representing `source AS alias`.
 */
exports.src = src;
const srcAs = (index, alias) => {
  const source = src(index);
  const aliasNode = _builder.Builder.identifier({
    name: alias
  });
  const node = _builder.Builder.expression.func.binary('as', [source, aliasNode]);
  return _synth_node.SynthNode.from(node);
};
/**
 * Crates an ES|QL integer literal node.
 *
 * @param value The integer value to create a literal for.
 * @returns ES|QL integer literal node.
 */
exports.srcAs = srcAs;
const int = value => {
  const node = _builder.Builder.expression.literal.integer(value);
  return _synth_node.SynthNode.from(node);
};

/**
 * Creates an ES|QL decimal (float, double) literal node.
 *
 * @param value The decimal value to create a literal for.
 * @returns ES|QL decimal literal node.
 */
exports.int = int;
const float = value => {
  const node = _builder.Builder.expression.literal.decimal(value);
  return _synth_node.SynthNode.from(node);
};

/**
 * Creates an ES|QL numeric literal node. If the value is an integer, an integer
 * literal node is created, otherwise a decimal literal node is created.
 *
 * @param value The numeric value to create a literal for.
 * @returns ES|QL numeric literal node.
 */
exports.float = float;
const num = value => {
  if (Number.isInteger(value)) {
    return int(value);
  } else {
    return float(value);
  }
};

/**
 * Creates an ES|QL boolean literal node.
 *
 * @param value The boolean value to create a literal for.
 * @returns ES|QL boolean literal node.
 */
exports.num = num;
const bool = value => {
  const node = _builder.Builder.expression.literal.boolean(value);
  return _synth_node.SynthNode.from(node);
};

/**
 * Creates an ES|QL string literal node.
 *
 * @param value The string value to create a literal for.
 * @returns ES|QL string literal node.
 */
exports.bool = bool;
const str = value => {
  const node = _builder.Builder.expression.literal.string(value);
  return _synth_node.SynthNode.from(node);
};

/**
 * Creates an ES|QL named parameter node.
 *
 * @param name The name of the parameter.
 * @returns ES|QL named parameter node.
 */
exports.str = str;
const par = name => {
  const node = _builder.Builder.param.build(name);
  return _synth_node.SynthNode.from(node);
};

/**
 * Creates an ES|QL named double parameter node.
 *
 * @param name The name of the parameter.
 * @returns ES|QL named parameter node.
 */
exports.par = par;
const dpar = name => {
  const node = _builder.Builder.param.named({
    value: name,
    paramKind: '??'
  });
  return _synth_node.SynthNode.from(node);
};

/**
 * Creates an ES|QL column node.
 *
 * @param name The name of the column.
 * @returns ES|QL column node.
 */
exports.dpar = dpar;
const col = name => {
  let columnName;
  let qualifierName;

  // Identify qualified columns names tuples ['qualifier', ['fieldName']]
  if ((0, _holes.isQualifiedColumnShorthand)(name)) {
    qualifierName = name[0];
    columnName = name[1];
  } else {
    // Simple or nested column
    columnName = name;
  }
  const node = _builder.Builder.expression.column(columnName, qualifierName);
  return _synth_node.SynthNode.from(node);
};

/**
 * Creates a literal fragment representing a keyword.
 *
 * @param keyword The keyword to create a literal fragment for.
 * @returns A SynthLiteralFragment representing the keyword.
 */
exports.col = col;
const kwd = keyword => {
  return new _synth_literal_fragment.SynthLiteralFragment(keyword);
};
exports.kwd = kwd;