"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CompletionPosition = void 0;
exports.autocomplete = autocomplete;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _is = require("../../../ast/is");
var _utils = require("../../definitions/utils");
var _map_expression = require("../../definitions/utils/autocomplete/map_expression");
var _constants = require("../../definitions/constants");
var _complete_items = require("../complete_items");
var _helpers = require("../../definitions/utils/autocomplete/helpers");
var _literals = require("../../definitions/utils/literals");
var _types = require("../types");
var _functions = require("../../definitions/utils/functions");
var _types2 = require("../../../shared/sorting/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
let CompletionPosition = exports.CompletionPosition = /*#__PURE__*/function (CompletionPosition) {
  CompletionPosition["AFTER_COMPLETION"] = "after_completion";
  CompletionPosition["AFTER_TARGET_FIELD"] = "after_target_field";
  CompletionPosition["AFTER_TARGET_ASSIGNMENT"] = "after_target_assignment";
  CompletionPosition["EXPRESSION"] = "expression";
  CompletionPosition["AFTER_WITH_KEYWORD"] = "after_with_keyword";
  CompletionPosition["WITHIN_MAP_EXPRESSION"] = "within_map_expression";
  CompletionPosition["AFTER_COMMAND"] = "after_command";
  return CompletionPosition;
}({});
function getPosition(query, command, isExistingColumn) {
  var _paramsMap;
  const {
    prompt,
    targetField
  } = command;
  const arg1 = command.args[1];
  let paramsMap;
  if (arg1 && 'type' in arg1 && arg1.type === 'option') {
    paramsMap = arg1.args[0];
    if (paramsMap && paramsMap.incomplete && !paramsMap.text) {
      return {
        position: CompletionPosition.AFTER_WITH_KEYWORD
      };
    }
  }
  if ((_paramsMap = paramsMap) !== null && _paramsMap !== void 0 && _paramsMap.text && paramsMap.incomplete) {
    return {
      position: CompletionPosition.WITHIN_MAP_EXPRESSION
    };
  }
  if (paramsMap && !paramsMap.incomplete) {
    return {
      position: CompletionPosition.AFTER_COMMAND
    };
  }
  const expressionRoot = (prompt === null || prompt === void 0 ? void 0 : prompt.text) !== _constants.EDITOR_MARKER ? prompt : undefined;

  // (function, literal, or existing column) - handle as primaryExpression
  if ((0, _is.isFunctionExpression)(expressionRoot) || (0, _is.isLiteral)(prompt) || isExistingColumn) {
    return {
      position: CompletionPosition.EXPRESSION,
      expressionRoot
    };
  }
  if (targetField && !targetField.incomplete) {
    return {
      position: CompletionPosition.AFTER_TARGET_ASSIGNMENT
    };
  }
  if ((prompt === null || prompt === void 0 ? void 0 : prompt.type) === 'column' && query.endsWith(' ')) {
    return {
      position: CompletionPosition.AFTER_TARGET_FIELD
    };
  }
  return {
    position: CompletionPosition.AFTER_COMPLETION
  };
}
const promptText = 'Your prompt to the LLM.';
const promptSnippetText = `"$\{0:${promptText}}"`;
async function autocomplete(query, command, callbacks, context, cursorPosition = query.length) {
  const innerText = query.substring(0, cursorPosition);
  const {
    prompt
  } = command;
  const isExistingColumn = (0, _helpers.columnExists)(prompt === null || prompt === void 0 ? void 0 : prompt.text, context);
  const {
    position,
    expressionRoot
  } = getPosition(innerText, command, isExistingColumn);
  if (!(callbacks !== null && callbacks !== void 0 && callbacks.getByType)) {
    return [];
  }
  const endpoints = context === null || context === void 0 ? void 0 : context.inferenceEndpoints;
  switch (position) {
    case CompletionPosition.AFTER_COMPLETION:
    case CompletionPosition.AFTER_TARGET_ASSIGNMENT:
      {
        const types = ['text', 'keyword', 'unknown'];
        const allSuggestions = [];

        // Fields
        allSuggestions.push(...(await (0, _utils.getFieldsSuggestions)(types, callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType, {
          ignoreColumns: [],
          values: false,
          addSpaceAfterField: false,
          openSuggestions: false,
          promoteToTop: true
        })));

        // Date literals (policy-gated in helpers) with explicit UI options
        allSuggestions.push(...(0, _utils.getLiteralsSuggestions)(types, _types.Location.COMPLETION, {
          includeDateLiterals: true,
          includeCompatibleLiterals: false,
          addComma: false,
          advanceCursorAndOpenSuggestions: false
        }));

        // Functions
        allSuggestions.push(...(0, _utils.getFunctionsSuggestions)({
          location: _types.Location.COMPLETION,
          types,
          options: {
            ignored: []
          },
          context,
          callbacks
        }));
        const fieldsAndFunctionsSuggestions = (0, _lodash.uniqBy)(allSuggestions, 'label');
        const suggestions = await (0, _helpers.handleFragment)(innerText, fragment => Boolean((0, _helpers.columnExists)(fragment, context) || (0, _functions.getFunctionDefinition)(fragment)), (_fragment, rangeToReplace) => {
          return fieldsAndFunctionsSuggestions.map(suggestion => {
            return (0, _helpers.withAutoSuggest)({
              ...suggestion,
              text: `${suggestion.text} `,
              rangeToReplace
            });
          });
        }, () => []);
        const lastWord = (0, _helpers.findFinalWord)(innerText);
        if (!lastWord) {
          suggestions.push({
            ...(0, _literals.buildConstantsDefinitions)([promptSnippetText], '', '1', undefined, undefined, _types2.SuggestionCategory.CONSTANT_VALUE)[0],
            label: promptText,
            asSnippet: true
          });
        }
        if (position !== CompletionPosition.AFTER_TARGET_ASSIGNMENT) {
          var _callbacks$getSuggest;
          suggestions.push((0, _complete_items.getNewUserDefinedColumnSuggestion)((callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getSuggest = callbacks.getSuggestedUserDefinedColumnName) === null || _callbacks$getSuggest === void 0 ? void 0 : _callbacks$getSuggest.call(callbacks)) || ''));
        }
        return suggestions;
      }
    case CompletionPosition.AFTER_TARGET_FIELD:
      return [_complete_items.assignCompletionItem];
    case CompletionPosition.EXPRESSION:
      {
        if ((0, _helpers.withinQuotes)(innerText)) {
          return [];
        }
        if (!(0, _is.isLiteral)(prompt)) {
          const {
            suggestions,
            computed
          } = await (0, _utils.suggestForExpression)({
            query,
            expressionRoot,
            command,
            cursorPosition,
            location: _types.Location.COMPLETION,
            context,
            callbacks
          });
          if (computed.insideFunction || !computed.isComplete) {
            return suggestions;
          }
        }
        return [{
          ..._complete_items.withCompleteItem,
          detail: _i18n.i18n.translate('kbn-esql-ast.esql.definitions.completionWithDoc', {
            defaultMessage: 'Provide additional parameters for the LLM prompt.'
          })
        }];
      }
    case CompletionPosition.AFTER_WITH_KEYWORD:
      return [_complete_items.withMapCompleteItem];
    case CompletionPosition.WITHIN_MAP_EXPRESSION:
      const availableParameters = {
        inference_id: {
          type: 'string',
          suggestions: (endpoints === null || endpoints === void 0 ? void 0 : endpoints.map(_helpers.createInferenceEndpointToCompletionItem)) || []
        }
      };
      return (0, _map_expression.getCommandMapExpressionSuggestions)(innerText, availableParameters);
    case CompletionPosition.AFTER_COMMAND:
      return [_complete_items.pipeCompleteItem];
    default:
      return [];
  }
}