"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.acceptsArbitraryExpressions = acceptsArbitraryExpressions;
exports.buildExpressionFunctionParameterContext = buildExpressionFunctionParameterContext;
exports.getRightmostNonVariadicOperator = getRightmostNonVariadicOperator;
exports.isInOperator = isInOperator;
exports.isNullCheckOperator = isNullCheckOperator;
exports.isSpecialOperator = isSpecialOperator;
exports.matchesSpecialFunction = matchesSpecialFunction;
exports.specialOperators = void 0;
var _all_operators = require("../../../all_operators");
var _ = require("../..");
var _signature_analyzer = require("./signature_analyzer");
var _ast = require("../../../../../ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** IN, NOT IN, IS NULL, IS NOT NULL operators requiring special autocomplete handling */
const specialOperators = exports.specialOperators = [..._all_operators.inOperators, ..._all_operators.nullCheckOperators];

/**
 * Detects if function signatures accept arbitrary/complex expressions in parameters.
 *
 * This pattern indicates functions where parameters can contain complex expressions
 * (not just simple values), characterized by:
 * - Variadic with multiple parameters (minParams >= 2)
 * - Unknown return type (depends on arguments)
 * - Mixed parameter types (boolean + any)
 *
 * Examples: CASE(condition1, value1, condition2, value2, ..., default)
 */
function acceptsArbitraryExpressions(signatures) {
  if (!signatures || signatures.length === 0) {
    return false;
  }
  return signatures.some(sig => {
    const isVariadicWithMultipleParams = sig.minParams != null && sig.minParams >= 2;
    const hasUnknownReturn = sig.returnType === 'unknown';
    const hasMixedBooleanAndAny = sig.params.some(({
      type
    }) => type === 'boolean') && sig.params.some(p => p.type === 'any');
    return isVariadicWithMultipleParams && hasUnknownReturn && hasMixedBooleanAndAny;
  });
}

/** Checks if operator is a NULL check (IS NULL, IS NOT NULL) */
function isNullCheckOperator(name) {
  const upperName = name.toUpperCase();
  return _all_operators.nullCheckOperators.some(op => op.name.toUpperCase() === upperName);
}

/** Checks if operator is IN or NOT IN */
function isInOperator(name) {
  const lowerName = name.toLowerCase();
  return lowerName === 'in' || lowerName === 'not in';
}

/** Checks if operator requires special handling */
function isSpecialOperator(name) {
  const lowerName = name.toLowerCase();
  return specialOperators.some(op => op.name.toLowerCase() === lowerName);
}

/** Checks if function name matches a special function (case-insensitive) */
function matchesSpecialFunction(name, expected) {
  return name.toLowerCase() === expected;
}

/**
 *   Builds function parameter context for suggestions
 *   Commands with special filtering (like STATS) can extend with command-specific functionsToIgnore
 */
function buildExpressionFunctionParameterContext(fn, context) {
  const fnDefinition = (0, _.getFunctionDefinition)(fn.name);
  if (!fnDefinition || !context) {
    return null;
  }
  const analyzer = _signature_analyzer.SignatureAnalyzer.fromNode(fn, context, fnDefinition);
  if (!analyzer) {
    return null;
  }
  return {
    paramDefinitions: analyzer.getCompatibleParamDefs(),
    functionsToIgnore: [fn.name],
    // Basic recursion prevention
    hasMoreMandatoryArgs: analyzer.getHasMoreMandatoryArgs(),
    functionDefinition: fnDefinition,
    firstArgumentType: analyzer.getFirstArgumentType(),
    firstValueType: analyzer.getFirstValueType(),
    currentParameterIndex: analyzer.getCurrentParameterIndex(),
    validSignatures: analyzer.getValidSignatures()
  };
}

/**
 * Finds the rightmost non-variadic operator in an expression tree.
 * Useful for locating the most specific node near the cursor.
 */
function getRightmostNonVariadicOperator(root) {
  if ((root === null || root === void 0 ? void 0 : root.type) !== 'function') {
    return root;
  }
  let rightmostFn = root;
  const walker = new _ast.Walker({
    visitFunction: fn => {
      if (fn.subtype !== 'variadic-call' && fn.location.min > rightmostFn.location.min) {
        rightmostFn = fn;
      }
    }
  });
  walker.walkFunction(root);
  return rightmostFn;
}