"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.shouldSuggestComma = shouldSuggestComma;
var _types = require("../../../types");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const positionHandlers = {
  after_complete: handleAfterComplete,
  empty_expression: handleEmptyOrEnumExpression,
  enum_value: handleEmptyOrEnumExpression,
  inside_list: handleInsideList
};
function shouldSuggestComma(context) {
  const {
    position,
    isCursorFollowedByComma = false
  } = context;

  // Never suggest comma if cursor is already followed by one
  if (isCursorFollowedByComma) {
    return false;
  }
  const isExpressionHeavyFunction = hasExpressionHeavyParameters(context.functionSignatures);
  const handler = positionHandlers[position];
  if (!handler) {
    return false;
  }
  return handler(context, isExpressionHeavyFunction);
}
function handleAfterComplete(context, isExpressionHeavy) {
  const {
    typeMatches,
    hasMoreParams,
    isVariadic,
    isAmbiguousPosition,
    expressionType
  } = context;

  // Repeating signatures (eg. CASE): positions 2, 4, 6... are ambiguous
  // - If expression is boolean → it's a condition → suggest comma for next value
  // - If expression is NOT boolean → could be default value → no comma
  if (isAmbiguousPosition && expressionType !== 'boolean') {
    return false;
  }
  if (!hasMoreParams) {
    return false;
  }
  if (isExpressionHeavy && !typeMatches) {
    return false;
  }
  if (!typeMatches && !isVariadic) {
    return false;
  }
  return true;
}
function handleEmptyOrEnumExpression(context, isExpressionHeavy) {
  const {
    hasMoreMandatoryArgs = false,
    functionType
  } = context;

  // No more mandatory args: no comma
  if (!hasMoreMandatoryArgs) {
    return false;
  }

  // Operators don't get commas
  if (functionType === _types.FunctionDefinitionTypes.OPERATOR) {
    return false;
  }

  // Expression-heavy functions (e.g., CASE): no auto-comma
  // User will add operators/expressions before moving to next param
  if (isExpressionHeavy) {
    return false;
  }
  return true;
}
function handleInsideList(context) {
  const {
    listHasValues = false,
    innerText = ''
  } = context;

  // No values in list yet: no comma
  if (!listHasValues) {
    return false;
  }

  // Already after comma: no comma
  const lastChar = innerText.trimEnd().slice(-1);
  if (lastChar === ',') {
    return false;
  }
  return true;
}
function hasExpressionHeavyParameters(signatures) {
  if (!signatures || signatures.length === 0) {
    return false;
  }

  // Pattern: boolean + any params, minParams >= 2, returnType = unknown
  // Example: CASE function accepts expressions in all parameters
  return (0, _utils.acceptsArbitraryExpressions)(signatures);
}