"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findAttackDiscoveriesRoute = void 0;
var _actions = require("@kbn/security-solution-features/actions");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _zodHelpers = require("@kbn/zod-helpers");
var _helpers = require("../../../helpers");
var _build_response = require("../../../../lib/build_response");
var _index_privileges = require("../../helpers/index_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const findAttackDiscoveriesRoute = router => {
  router.versioned.get({
    access: 'public',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_FIND,
    security: {
      authz: {
        requiredPrivileges: [_actions.ATTACK_DISCOVERY_API_ACTION_ALL]
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryFindRequestQuery)
      },
      response: {
        200: {
          body: {
            custom: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryFindResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Perform license and authenticated user checks:
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }

    // Perform alerts access check
    const privilegesCheckResponse = await (0, _index_privileges.hasReadAttackDiscoveryAlertsPrivileges)({
      context: ctx,
      response
    });
    if (!privilegesCheckResponse.isSuccess) {
      return privilegesCheckResponse.response;
    }
    try {
      var _query$include_unique, _query$enable_field_r, _query$with_replaceme;
      const {
        query
      } = request;
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const currentUser = await checkResponse.currentUser;

      // get an Elasticsearch client for the authenticated user:
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const result = await dataClient.findAttackDiscoveryAlerts({
        authenticatedUser: currentUser,
        esClient,
        findAttackDiscoveryAlertsParams: {
          alertIds: query.alert_ids,
          includeUniqueAlertIds: (_query$include_unique = query.include_unique_alert_ids) !== null && _query$include_unique !== void 0 ? _query$include_unique : false,
          ids: query.ids,
          search: query.search,
          shared: query.shared,
          status: query.status,
          connectorNames: query.connector_names,
          start: query.start,
          enableFieldRendering: (_query$enable_field_r = query.enable_field_rendering) !== null && _query$enable_field_r !== void 0 ? _query$enable_field_r : false,
          // default to false for public APIs
          end: query.end,
          page: query.page,
          perPage: query.per_page,
          sortField: query.sort_field,
          sortOrder: query.sort_order,
          withReplacements: (_query$with_replaceme = query.with_replacements) !== null && _query$with_replaceme !== void 0 ? _query$with_replaceme : true // public APIs default to applying replacements in responses as a convenience to non-Kibana clients
        },
        logger
      });
      return response.ok({
        body: {
          ...result
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.findAttackDiscoveriesRoute = findAttackDiscoveriesRoute;