"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToCreateScheme = exports.createConversation = void 0;
var _uuid = require("uuid");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _get_conversation = require("./get_conversation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createConversation = async ({
  esClient,
  conversationIndex,
  spaceId,
  user,
  conversation,
  logger
}) => {
  const createdAt = new Date().toISOString();
  const document = transformToCreateScheme(createdAt, spaceId, user, conversation);
  try {
    const response = await esClient.create({
      document,
      id: (conversation === null || conversation === void 0 ? void 0 : conversation.id) || (0, _uuid.v4)(),
      index: conversationIndex,
      refresh: 'wait_for'
    });
    const createdConversation = await (0, _get_conversation.getConversation)({
      esClient,
      conversationIndex,
      id: response._id,
      logger,
      user
    });
    return createdConversation;
  } catch (err) {
    logger.error(`Error creating conversation: ${err} with title: ${conversation.title}`);
    throw err;
  }
};
exports.createConversation = createConversation;
const transformToCreateScheme = (createdAt, spaceId, user, {
  title,
  apiConfig,
  category,
  excludeFromLastConversationStorage,
  messages,
  replacements
}) => {
  return {
    '@timestamp': createdAt,
    created_at: createdAt,
    created_by: {
      id: user.profile_uid,
      name: user.username
    },
    users: [{
      id: user.profile_uid,
      name: user.username
    }],
    title,
    category: category !== null && category !== void 0 ? category : _elasticAssistantCommon.ConversationCategoryEnum.assistant,
    api_config: apiConfig ? {
      action_type_id: apiConfig.actionTypeId,
      connector_id: apiConfig.connectorId,
      default_system_prompt_id: apiConfig.defaultSystemPromptId,
      model: apiConfig.model,
      provider: apiConfig.provider
    } : undefined,
    exclude_from_last_conversation_storage: excludeFromLastConversationStorage,
    messages: messages === null || messages === void 0 ? void 0 : messages.map(message => {
      var _message$id;
      return {
        '@timestamp': message.timestamp,
        id: (_message$id = message.id) !== null && _message$id !== void 0 ? _message$id : (0, _uuid.v4)(),
        content: message.content,
        ...(message.refusal ? {
          refusal: message.refusal
        } : {}),
        is_error: message.isError,
        reader: message.reader,
        role: message.role,
        user: message.user,
        ...(message.traceData ? {
          trace_data: {
            trace_id: message.traceData.traceId,
            transaction_id: message.traceData.transactionId
          }
        } : {}),
        ...(message.metadata ? {
          metadata: {
            content_references: message.metadata.contentReferences
          }
        } : {})
      };
    }),
    updated_at: createdAt,
    replacements: replacements ? Object.keys(replacements).map(key => ({
      uuid: key,
      value: replacements[key]
    })) : undefined,
    namespace: spaceId
  };
};
exports.transformToCreateScheme = transformToCreateScheme;